#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of the csp-lmc-prototype project
#
#
#
# Distributed under the terms of the BSD-3-Clause license.
# See LICENSE.txt for more info.
"""Contain the tests for the CspMaster."""

# Standard imports
import sys
import os
import time
import logging
import numpy as np
# Tango imports
import tango
from tango import DevState
#from tango.test_context import DeviceTestContext
import pytest

# Path
file_path = os.path.dirname(os.path.abspath(__file__))
# insert base package directory to import global_enum
# module in commons folder
commons_pkg_path = os.path.abspath(os.path.join(file_path, "./csp-lmc-mid"))
sys.path.insert(0, commons_pkg_path)

path = os.path.join(os.path.dirname(__file__), os.pardir)
sys.path.insert(0, os.path.abspath(path))
print(commons_pkg_path)
#Local imports
from ska.base.control_model import AdminMode

LOGGER = logging.getLogger(__name__)
# Device test case
@pytest.mark.usefixtures("cbf_master", "midcsp_master", "cbf_subarray01", "midcsp_subarray01", "midcsp_subarray02")

class TestCspMaster(object):

    def test_State(self, midcsp_master, cbf_master):
        """Test for State after initialization """
        # reinitalize Csp Master and CbfMaster devices
        cbf_master.Init()
        time.sleep(2)
        midcsp_master.Init()
        time.sleep(2)
        csp_state = midcsp_master.state()
        assert csp_state in [DevState.STANDBY, DevState.INIT, DevState.DISABLE]

    def test_adminMode(self, midcsp_master):
        """ Test the adminMode attribute w/r"""
        midcsp_master.adminMode = AdminMode.OFFLINE
        time.sleep(3)
        assert midcsp_master.adminMode == AdminMode.OFFLINE
        assert midcsp_master.cspCbfAdminMode == AdminMode.OFFLINE
        midcsp_master.adminMode = AdminMode.ONLINE
        time.sleep(3)
        assert midcsp_master.adminMode == AdminMode.ONLINE
        assert midcsp_master.cspCbfAdminMode == AdminMode.ONLINE

    def test_properties(self, midcsp_master):
        """ Test the device property MaxCapability"""
        capability_list = ['FSP:27', 'VCC:4','SearchBeams:1500', 'TimingBeams:16', 'VlbiBeams:20']
        capability_list.sort()
        #Oss: maxCapability returns a tuple
        assert midcsp_master.maxCapabilities == tuple(capability_list)

    def test_forwarded_attributes(self, midcsp_master, cbf_master):
        """ Test the  reportVCCState forwarded attribute"""
        vcc_state = midcsp_master.reportVCCState
        vcc_state_cbf = cbf_master.reportVCCState
        assert np.array_equal(vcc_state, vcc_state_cbf)

    def test_receptors_unassiged_ids(self, midcsp_master):
        """ Test the reading of availableReceptorIDs attribute """
        list_of_receptors = midcsp_master.unassignedReceptorIDs
        LOGGER.info("list of receptors:".format(list_of_receptors))
        assert any(list_of_receptors)

    def test_read_capabilities_addresses(self, midcsp_master):
        """ Test the reading of availableCapabilities attribute """
        vcc_addresses = midcsp_master.vccCapabilitiesAddresses
        assert vcc_addresses
        searchbeams_addresses = midcsp_master.searchBeamsAddresses
        assert searchbeams_addresses
        timingbeams_addresses = midcsp_master.timingBeamsAddresses
        assert timingbeams_addresses
        vlbi_addresses = midcsp_master.vlbiBeamsAddresses
        assert vlbi_addresses

    def test_available_capabilities(self, midcsp_master, ):
        """ Test the reading of availableCapabilities attribute """
        available_cap = midcsp_master.availableCapabilities
        assert available_cap

    def test_On_valid_state(self, midcsp_master, cbf_master, midcsp_subarray01, midcsp_subarray02):
        """
        Test for execution of On command when the CbfMaster is in the right state
        """
        #reinit CSP and CBF master devices
        cbf_master.Init()
        time.sleep(2)
        # sleep for a while to wait state transition
        # check CspMaster state
        midcsp_master.Init()
        assert midcsp_master.State() == DevState.STANDBY
        # issue the "On" command on CbfMaster device
        argin = ["mid_csp_cbf/sub_elt/master",]
        midcsp_master.On(argin)
        time.sleep(3)
        LOGGER.info(" midcsp_subarray01.state:{}".format( midcsp_subarray01.state()))
        LOGGER.info(" midcsp_subarray02.state:{}".format( midcsp_subarray02.state()))
        assert midcsp_master.state() == DevState.ON
        assert midcsp_subarray01.state() == DevState.ON
        assert midcsp_subarray02.state() == DevState.ON

    def test_Standby_valid_state(self, midcsp_master, midcsp_subarray01, midcsp_subarray02):
        """
        Test for execution of On command when the CbfTestMaster is in the right state
        """
        assert midcsp_master.State() == DevState.ON
        # issue the "Standby" command on CbfMaster device
        argin = ["mid_csp_cbf/sub_elt/master",]
        midcsp_master.Standby(argin)
        time.sleep(3)
        assert midcsp_master.state() == DevState.STANDBY
        assert midcsp_subarray01.state() == DevState.OFF
        assert midcsp_subarray02.state() == DevState.OFF

    def test_reinit_midcsp_master(self, midcsp_master, cbf_master):
        """ Test CspMaster reinitialization """
        #reinit CSP and CBFTest master devices
        cbf_master.Init()
        time.sleep(2)
        midcsp_master.Init()
        time.sleep(2)
        assert midcsp_master.State() == DevState.STANDBY
