#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of the csp-lmc project
#
#
#
# Distributed under the terms of the BSD-3-Clause license.
# See LICENSE.txt for more info.
"""Tests for ADR-4 and ADR-10 implementation into CSP.LMC Subarray."""

# Standard imports
import sys
import os
import time
import logging
from pytest import fail

# Tango imports
import tango
from tango import DevState, EventType
import pytest
import unittest

# Local imports
from ska.base.control_model import ObsState
from utils import Poller, Probe
from assertpy import assert_that

# Path
file_path = os.path.dirname(os.path.abspath(__file__))
# insert base package directory to import global_enum
# module in commons folder
data_pkg_path = os.path.abspath(os.path.join(file_path, "./tests/acceptance_tests/"))
sys.path.insert(0, data_pkg_path)

path = os.path.join(os.path.dirname(__file__), os.pardir)
sys.path.insert(0, os.path.abspath(path))

LOGGER = logging.getLogger(__name__)


def prepare_configuration_string(filename="test_ConfigureScan_ADR4.json"):
    """Create the config string for CSP-CBF"""
    try:
        file_to_load = file_path + '/' + filename
        LOGGER.info(f"Reading configuration from {file_to_load}")
        json_file = open(file_to_load)
        configuration_string = json_file.read().replace("\n", "")
        return configuration_string
    except Exception as e:
        LOGGER.warning(f"Unable to locate file {filename}")


# Device test case
@pytest.mark.usefixtures("midcsp_master", "midcsp_subarray01", "cbf_subarray01")
class TestBase(unittest.TestCase):
    fixture_names = ()

    @pytest.fixture(autouse=True)
    def auto_injector_fixture(self, request):
        names = self.fixture_names
        for name in names:
            setattr(self, name, request.getfixturevalue(name))


class TestCspSubarrayConfiguration(TestBase):
    fixture_names = ("midcsp_master", "midcsp_subarray01", "cbf_subarray01")

    def __set_csp_subarray_to_on(self):
        """
        Set the CSP Subarray to ON state.
        """
        # read from the CSP Master the list of available receptors
        # and the receptorMembership attribute reporting
        # dish affiliation to subarrays
        unassigned_receptors = self.midcsp_master.unassignedReceptorIDs
        # assign the first one to the CSP subarray to force 
        # the transition to ON
        LOGGER.debug("Add receptors to CSP Subarray")
        self.midcsp_subarray01.AddReceptors([unassigned_receptors[0], ])
        prober_cmd_in_progress = Probe(self.midcsp_subarray01, 'isCmdInProgress', False,
                                           f"Wrong value for command in progress")
        Poller(10, 0.2).check(prober_cmd_in_progress)
        state = self.midcsp_subarray01.State()
        assert state == DevState.ON

    def _set_cspsubarray_to_idle_state(self):
        LOGGER.info("Set the CSP Subarray to IDLE")
        obs_state = self.midcsp_subarray01.obsState
        if obs_state == ObsState.READY:
            self.midcsp_subarray01.GoToIdle()
            prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.IDLE, f"CSP Subarray not IDLE")
            Poller(7, 0.2).check(prober_obs_state)

    def _reset_subarray_to_disable_state(self):
        try:
            state = self.midcsp_subarray01.State()
            LOGGER.debug("CSP Subarray init State:{}".format(state))
            if state == DevState.DISABLE:
                return
            if state == DevState.ON:
                self._set_cspsubarray_to_idle_state()
                LOGGER.debug("Remove all receptors")
                self.midcsp_subarray01.RemoveAllReceptors()
                # wait for the transition of the CSP sub-array to OFF
                prober_subarray_state = Probe(self.midcsp_subarray01, 'state', DevState.OFF,
                                              f"Wrong CSP Subarray state")
                Poller(7, 0.2).check(prober_subarray_state)

            # Set the CSP subarray to OFF issuing the Standby command
            # on CSP Master
            state = self.midcsp_subarray01.State()
            if state == DevState.OFF:
                LOGGER.debug("CSP Subarray State is OFF...going to execute Standby")
                # wait until Master is ON
                prober_master_state = Probe(self.midcsp_master, 'state', DevState.ON, f"Wrong CSP Master state")
                Poller(7, 0.2).check(prober_master_state)
                self.midcsp_master.Standby("")
                prober_subarray_state = Probe(self.midcsp_subarray01, 'state', DevState.DISABLE,
                                              f"Wrong CSP Subarray state")
                Poller(5, 0.2).check(prober_subarray_state)
                prober_master_state = Probe(self.midcsp_master, 'state', DevState.STANDBY,
                                            f"Wrong CSP Master state")
                Poller(7, 0.2).check(prober_master_state)

        except tango.DevFailed as tango_err:
            LOGGER.warning(f"Unable to reset subarray to init state")

    def _setup_csp_subarray(self):
        """
        Set the CSP Subarray state to ON
        """
        LOGGER.info("Setup subarray")
        self._reset_subarray_to_disable_state()
        # switch-on the CspMaster to force CSP Subarray
        # transition to OFF
        LOGGER.debug("Set CSP Subarray to OFF")
        self.midcsp_master.On("")
        # wait for the transition of the CSP subarray to OFF
        prober_subarray_state = Probe(self.midcsp_subarray01, 'state', DevState.OFF, f"Wrong CSP Subarray state")
        Poller(10, 0.2).check(prober_subarray_state)
        state = self.midcsp_subarray01.State()
        assert state in [DevState.OFF], "assuming that mid_csp_subarray_01 is OFF"
        self.__set_csp_subarray_to_on()

    def test_send_configure_to_cbf_and_json_stored(self):
        """
        Configure the CSP Subarray with a JSon string including
        the new ADR4 fields.
        """
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string()
        state = self.midcsp_subarray01.State()
        assert state == DevState.ON, "assume the CSP subarray State is ON"
        # exercise the device
        LOGGER.debug(f"sending configure:{configuration_string}")
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(configuration_string)
        prober_subarray_obstate = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY,
                                        f"Wrong CSP Subarray obsState {self.midcsp_subarray01.obsState}")
        Poller(5, 0.2).check(prober_subarray_obstate)
        # check
        #obs_state = self.midcsp_subarray01.obsState
        #assert obs_state == ObsState.READY,f"Wrong CSP Subarray obsState {obs_state}"
        stored_json = self.midcsp_subarray01.validScanConfiguration
        assert stored_json == configuration_string

    def test_configure_cspsubarray_WITH_json_missing_configId(self):
        """
        Configure the CSP Subarray sending a json configuration script 
        without the configID entry. 
        """
        # setup the test
        # Set the CSP Subarray to ON-IDLE state
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string("test_ConfigureScan_without_configID.json")
        #configuration_string = prepare_configuration_string("test_ConfigureScan_ADR4.json")

        # exercise the device
        LOGGER.info(f"Configuring CSP subarray01")
        with pytest.raises(tango.DevFailed) as tango_err:
            self.midcsp_subarray01.Configure(configuration_string)
        if tango_err:
            err_msg = str(tango_err.value.args[0].desc)
            # TODO: aggiungere quale device fallisce
            LOGGER.warning(f"Command configure failed on {self.midcsp_subarray01.dev_name()} with error: {err_msg}")
        # check
        obs_state = self.midcsp_subarray01.obsState
        #assert obs_state == ObsState.IDLE, f"CSP Subarray obsState has wrong value ({obs_state})"
        assert_that(obs_state).described_as("CSP Subarray obsState has wrong value ({obs_state}").is_equal_to(ObsState.IDLE)

    def test_configure_cspsubarray_WHEN_in_wrong_state(self):
        """
        Set the CSP Subarray in a wrong state and issue the
        Configure command on it.
        """
        # setup the test: Subarray DISABLE-IDLE

        self._setup_csp_subarray()
        LOGGER.info("Force CSP Subarray to an invalid state to accept Configure")
        self._reset_subarray_to_disable_state()
        configuration_string = prepare_configuration_string()
        init_state = self.midcsp_subarray01.State()

        # exercise the device
        LOGGER.info(f"Configuring CSP subarray01")
        with pytest.raises(tango.DevFailed) as df:
            self.midcsp_subarray01.Configure(configuration_string)
        #assert_that(self.midcsp_subarray01.Configure).raises(tango.DevFailed).when_called_with(configuration_string)
        if df:
            err_msg = str(df.value.args[0].desc)
            # TODO: aggiungere quale device fallisce
            LOGGER.error(f"Command configure failed on {self.midcsp_subarray01.dev_name()} with error: {err_msg}")
        # check
        # Subarray final ObsState=IDLE
        obs_state = self.midcsp_subarray01.obsState
        end_state = self.midcsp_subarray01.State()
        assert obs_state == ObsState.IDLE, f"Current ObsState should be IDLE"
        assert end_state == init_state, f"Current State should be the initial one"
        #assert_that(obs_state).described_by("Current ObsState should be IDLE").is_not_equal_to(ObsState.IDLE)

    def test_configure_cbf_WITH_invalid_json(self):
        """
        CSP Subarray sends an invalid json configuration to
        CBF Subarray.
        """
        # setup the test: Subarray DISABLE-IDLE
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string("test_ConfigureScan_invalid_cbf_json.json")
        # exercise the device
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(configuration_string)
        # check
        # Subarray final ObsState IDLE
        prober_subarray_flag = Probe(self.midcsp_subarray01, 'failureRaisedFlag', True, f"Failure flag is false")
        Poller(7, 0.2).check(prober_subarray_flag)
        # failure_flag = self.midcsp_subarray01.failureRaisedFlag
        # assert failure_flag

    def test_cspsubarray_scan_WITH_invalid_string_id(self):
        """
        CSP Subarray receives the Scan command with an invalid 
        scanId number
        """
        # setup the test: Subarray DISABLE-IDLE
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string("test_ConfigureScan_ADR4.json")
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(configuration_string)
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY, f"CSP Subarray not configured")
        Poller(7, 0.2).check(prober_obs_state)
        LOGGER.info(f"Issue the Scan command sending the invalid string \"abcd\" as input")
        # exercise the device
        with pytest.raises(tango.DevFailed) as df:
            self.midcsp_subarray01.Scan("abcd")
        if df:
            err_msg = str(df.value.args[0].desc)
            LOGGER.error(f"Command Scan failed on {self.midcsp_subarray01.dev_name()} with error: {err_msg}")
        # check
        obs_state =  self.midcsp_subarray01.obsState
        assert obs_state == ObsState.READY

    def test_cspsubarray_scan_WITH_no_argument(self):
        """
        CSP Subarray receives the Scan command with no
        input argument
        """
        # setup the test: Subarray DISABLE-IDLE
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string("test_ConfigureScan_ADR4.json")
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(configuration_string)
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY, f"CSP Subarray not configured")
        Poller(7, 0.2).check(prober_obs_state)
        LOGGER.info(f"Issue the Scan command with no input argument")
        # exercise the device
        with pytest.raises(tango.DevFailed) as df:
            self.midcsp_subarray01.Scan()
        if df:
            err_msg = str(df.value.args[0].desc)
            LOGGER.error(f"Command configure failed on {self.midcsp_subarray01.dev_name()} with error: {err_msg}")
        # check
        obs_state =  self.midcsp_subarray01.obsState
        assert obs_state == ObsState.READY

    def test_cspsubarray_scan_WITH_int_argument(self):
        """
        CSP Subarray receives the Scan command with an invalid 
        argument data type
        """
        # setup the test: Subarray DISABLE-IDLE
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string("test_ConfigureScan_ADR4.json")
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(configuration_string)
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY, f"CSP Subarray not configured")
        Poller(7, 0.2).check(prober_obs_state)
        LOGGER.info(f"Issue the Scan command sending an int as input")
        # exercise the device
        with pytest.raises(tango.DevFailed) as df:
            self.midcsp_subarray01.Scan(1)
        if df:
            err_msg = str(df.value.args[0].desc)
            LOGGER.error(f"Command Scan failed on {self.midcsp_subarray01.dev_name()} with error: {err_msg}")
        # check
        obs_state =  self.midcsp_subarray01.obsState
        assert obs_state == ObsState.READY

    def test_cspsubarray_scan_WITH_valid_scan_id(self):
        """
        CSP Subarray receives the Scan command with a valid
        input argument
        """
        # setup the test: Subarray DISABLE-IDLE
        self._setup_csp_subarray()
        configuration_string = prepare_configuration_string("test_ConfigureScan_ADR4.json")
        LOGGER.info(f"Configuring CSP subarray01")
        self.midcsp_subarray01.Configure(configuration_string)
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY, f"CSP Subarray not configured")
        Poller(7, 0.2).check(prober_obs_state)
        LOGGER.info(f"Issue the Scan command")
        # exercise the device
        self.midcsp_subarray01.Scan("1")
        # check
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.SCANNING, f"CSP Subarray wrong obstate")
        Poller(7, 0.2).check(prober_obs_state)
        # end the scan
        self.midcsp_subarray01.EndScan()
        prober_obs_state = Probe(self.midcsp_subarray01, 'obsState', ObsState.READY, f"CSP Subarray wrong obstate")
        Poller(7, 0.2).check(prober_obs_state)
