#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of the csp-lmc project
#
#
#
# Distributed under the terms of the BSD-3-Clause license.
# See LICENSE.txt for more info.
"""Contain the tests for the CspMaster."""

# Standard imports
import sys
import os
import time
import pytest
# Tango imports
import tango
from tango import DevState
from tango.test_context import DeviceTestContext
#Local imports
from csp_lmc_common.CspSubarray import CspSubarray
from ska.base.control_model import AdminMode, ObsState, HealthState

# Device test case
@pytest.mark.usefixtures("tango_context", "csp_master")

class TestCspSubarray(object):
    properties = {'CspMaster': 'common/elt/master',
                  'CbfSubarray': 'mid_csp_cbf/sub_elt/subarray_01',
                  'PssSubarray': 'mid_csp_pss/sub_elt/subarray_01',
                  'SubID': 1,
                }

    @classmethod
    def mocking(cls):
        """Mock external libraries."""
        # Example : Mock numpy
        # cls.numpy = SKASubarray.numpy = MagicMock()
        # PROTECTED REGION ID(SKASubarray.test_mocking) ENABLED START #
        # PROTECTED REGION END #    //  SKASubarray.test_mocking

    def test_init_State(self, tango_context):
        """Test for State at initialization """
        state = tango_context.device.state()
        assert state not in [DevState.FAULT]

    def test_cbfsubarray_state(self, tango_context):
        assert tango_context.device.cbfSubarrayState == DevState.DISABLE

    def test_cbfsubarray_adminMode(self, tango_context):
        assert tango_context.device.cbfSubarrayAdminMode == AdminMode.ONLINE

    def test_cbfsubarray_healthState(self, tango_context):
        assert tango_context.device.cbfSubarrayHealthState == HealthState.OK

    def test_cbfsubarray_obsState(self, tango_context):
        assert tango_context.device.cbfSubarrayObsState == ObsState.IDLE

    def test_psssubarray_state(self, tango_context):
        assert tango_context.device.pssSubarrayState == DevState.DISABLE

    def test_psssubarray_adminMode(self, tango_context):
        assert tango_context.device.pssSubarrayAdminMode == AdminMode.NOT_FITTED

    def test_psssubarray_healthState(self, tango_context):
        assert tango_context.device.pssSubarrayHealthState == HealthState.UNKNOWN

    def test_psssubarray_obsState(self, tango_context):
        assert tango_context.device.pssSubarrayObsState == ObsState.IDLE

    def test_init_obsState(self, tango_context):
        """Test the obsState value at initialization """
        obs_state = tango_context.device.obsState
        assert obs_state == ObsState.IDLE

    def test_commands_progress(self, tango_context):
        """Test xxCmdProgress attributes """
        assert tango_context.device.scanCmdProgress == 0
        assert tango_context.device.endScanCmdProgress == 0
        assert tango_context.device.goToIdleCmdProgress == 0

    def test_configure_invalid_state(self, tango_context):
        """ Test Configure command execution when subarray
            State is not ON
        """
        with pytest.raises(tango.DevFailed) as df:
            tango_context.device.Configure("")
        assert "Command Configure not allowed" in df.value.args[0].desc

    def test_subarray_off(self, tango_context, csp_master):
        """ Test subarray state after power-on
            NOTE: this is a wrong behaviour of CbfSubarray that
            starts in DISABLE state with adminMode = ONLINE and
            it moves to OFF only after a power-on.
            The CbfSubarray has to start in OFF State when it is
            ONLINE and there is no need to power-on it.
            This behavior has to be changed
        """
        csp_master.On(["mid_csp_cbf/sub_elt/master",])
        time.sleep(4)
        csp_state = csp_master.State()
        state = tango_context.device.cbfSubarrayState
        assert state == DevState.OFF

    def test_remove_timing_beams_while_add_is_running_state(self, tango_context):
        """ Test the execution of the RemovingTimingBeams" while
            the AddTimingBeams is already running
        """
        tango_context.device.AddTimingBeams([1,2,3])
        with pytest.raises(tango.DevFailed) as df:
            tango_context.device.RemoveTimingBeams([1,2])
        assert "Can't execute command" in df.value.args[0].desc
