#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of the csp-lmc project
#
#
#
# Distributed under the terms of the BSD-3-Clause license.
# See LICENSE.txt for more info.
"""Contain the tests for the CspMaster."""

# Standard imports
import sys
import os
import time
import numpy as np
# Tango imports
import tango
from tango import DevState
from tango.test_context import DeviceTestContext
import pytest

#Local imports
from csp_lmc_common.CspMaster import CspMaster
from ska.base.control_model import AdminMode, HealthState

# Device test case
@pytest.mark.usefixtures("tango_context", "cbf_master")

class TestCspMaster(object):
    properties = {'SkaLevel': '2', 
                  'CspCbf': 'mid_csp_cbf/sub_elt/master',
                  'CspPss': 'mid_csp_pss/sub_elt/master',
                  'CspPst': 'mid_csp_pst/sub_elt/master',
                  }

    def test_State(self, cbf_master, tango_context):
        """Test for State after initialization 
           NOTE: the CspMaster device state is ALARM because
           this device relies on several forwarded attributes.
        """
        # reinitalize Csp Master and CbfMaster devices
        cbf_master.Init()
        time.sleep(2)
        csp_state = tango_context.device.state()
        assert csp_state not in [DevState.FAULT, DevState.DISABLE]

    def test_adminMode_offline(self, tango_context):
        """ Test the adminMode attribute w/r"""
        tango_context.device.adminMode = AdminMode.OFFLINE
        time.sleep(3)
        assert tango_context.device.adminMode.value == AdminMode.OFFLINE

    def test_adminMode_online(self, tango_context):
        """ Test the adminMode attribute w/r"""
        tango_context.device.adminMode = AdminMode.ONLINE
        time.sleep(3)
        assert tango_context.device.adminMode.value == AdminMode.ONLINE

    def test_write_invalid_admin_mode(self, tango_context):

        with pytest.raises(tango.DevFailed) as df:
            tango_context.device.adminMode = 7
        assert "Set value for attribute adminMode is negative or above the maximun authorized" in str(df.value.args[0].desc)

    def test_cspHealthState(self, tango_context):
        """ Test the CSP initial healthState.
            NOTE: to build the CSP healthState, the sub-elements
            not ONLINE or MAINTENACE are not taken into account.
        """
        assert tango_context.device.healthState == HealthState.OK

    def test_cbfAdminMode(self, tango_context):
        """ Test the CBF adminMode attribute w/r"""
        tango_context.device.cspCbfAdminMode = AdminMode.ONLINE
        time.sleep(3)
        assert tango_context.device.cspCbfAdminMode.value == AdminMode.ONLINE

    def test_cbfHealthState(self, tango_context):
        """ Test the CBF initial healthState          """
        tango_context.device.cspCbfHealthState == HealthState.OK

    def test_pssState(self, tango_context):
        """ Test the PSS initial State          """
        assert tango_context.device.cspPssState == DevState.DISABLE

    def test_pssHealthState(self, tango_context):
        """ Test the PSS initial healthState          """
        assert tango_context.device.cspPssHealthState == HealthState.UNKNOWN

    def test_pssAdminMode(self, tango_context):
        """ Test the PSS initial adminMode          """
        pss_admin = tango_context.device.cspPssAdminMode
        assert pss_admin == AdminMode.NOT_FITTED

    def test_write_adminMode_of_not_running_device(self, tango_context):
        assert tango_context.device.cspPssAdminMode == AdminMode.NOT_FITTED
        tango_context.device.cspPssAdminMode == AdminMode.ONLINE
        time.sleep(3)
        assert tango_context.device.cspPssAdminMode == AdminMode.NOT_FITTED

    def test_pstState(self, tango_context):
        """ Test the PST initial State          """
        assert tango_context.device.cspPstState == DevState.DISABLE

    def test_pstHealthState(self, tango_context):
        """ Test the PST initial healthState          """
        assert tango_context.device.cspPstHealthState == HealthState.UNKNOWN

    def test_pstAdminMode(self, tango_context):
        """ Test the PST initial adminMode          """
        assert tango_context.device.cspPstAdminMode == AdminMode.NOT_FITTED

    def test_subelement_cbf_address(self, tango_context):
        """Test the cbfMasterAdress value"""
        cbf_addr = tango_context.device.cbfMasterAddress
        cbf_addr_property = self.properties['CspCbf']
        assert cbf_addr == cbf_addr_property

    def test_subelement_pss_address(self, tango_context):
        """Test the pssMasterAdress value"""
        pss_addr = tango_context.device.pssMasterAddress
        pss_addr_property = self.properties['CspPss']
        assert pss_addr == pss_addr_property

    def test_subelement_pst_address(self, tango_context):
        """Test the pstMasterAdress value"""
        pst_addr = tango_context.device.pstMasterAddress
        pst_addr_property = self.properties['CspPst']
        assert pst_addr == pst_addr_property

    def test_configure_On_command_duration_time(self, tango_context):
        tango_context.device.onCmdDurationExpected = 3
        assert tango_context.device.onCmdDurationExpected == 3

    def test_configure_Off_command_duration_time(self, tango_context):
        tango_context.device.offCmdDurationExpected = 3
        assert tango_context.device.offCmdDurationExpected == 3

    def test_configure_Standby_command_duration_time(self, tango_context):
        tango_context.device.standbyCmdDurationExpected = 3
        assert tango_context.device.standbyCmdDurationExpected == 3

    def test_On_valid_state(self, tango_context, cbf_master):
        """
        Test for execution of On command when the CbfTestMaster is in the right state
        """
        #reinit CSP and CBFTest master devices
        cbf_master.Init()
        # sleep for a while to wait state transition
        time.sleep(3)
        # check CspMstar state
        assert tango_context.device.cspCbfState == DevState.STANDBY
        # issue the "On" command on CbfTestMaster device
        argin = [tango_context.device.cbfMasterAddress,]
        tango_context.device.On(argin)
        time.sleep(3)
        assert tango_context.device.cspCbfState == DevState.ON

    def test_on_command_duration(self, tango_context):
        time_measured = tango_context.device.onCmdDurationMeasured
        assert time_measured > 0

    def test_on_execution_timeout(self, tango_context):
        on_timeout = tango_context.device.onCmdTimeoutExpired
        assert not on_timeout 
        time.sleep(2.5)
        assert tango_context.device.onCommandProgress == 100

    def test_read_list_of_completed_task(self, tango_context):
        list_of_task = tango_context.device.listOfDevCompletedTask
        print("list_of_task:", list_of_task)
        assert tango_context.device.cbfMasterAddress in list_of_task

    def test_Standby_valid_state(self, tango_context, cbf_master):
        assert tango_context.device.cspCbfState == DevState.ON
        # issue the "On" command on CbfTestMaster device
        argin = [tango_context.device.cbfMasterAddress,]
        tango_context.device.Standby(argin)
        time.sleep(3)
        assert tango_context.device.cspCbfState == DevState.STANDBY

    def test_standby_command_duration(self, tango_context):
        time_measured = tango_context.device.standbyCmdDurationMeasured
        assert time_measured > 0

    def test_standby_execution_timeout(self, tango_context):
        on_timeout = tango_context.device.standbyCmdTimeoutExpired
        assert not on_timeout 
        time.sleep(2.5)
        assert tango_context.device.standbyCommandProgress == 100

    def test_num_of_completed_task(self, tango_context):
        time.sleep(3)
        num_of_task = tango_context.device.numOfDevCompletedTask
        assert num_of_task == 1
