# -*- coding: utf-8 -*-
#
# This file is part of the MidCspSubarrayBase project
#
# INAF, SKA Telescope
#
# Distributed under the terms of the GPL license.
# See LICENSE.txt for more info.

""" MidCspSubarrayBase

The base class for MID CspSubarray.
Fuctionality to monitor assigned CSP.LMC Capabilities,
as well as inherent Capabilities, are implemented in 
separate TANGO Devices.
"""
# PROTECTED REGION ID (MidCspSubarrayBase.standardlibray_import) ENABLED START #
# Python standard library
import sys
import os
from future.utils import with_metaclass
from collections import defaultdict
import threading
import time

import json
# PROTECTED REGION END# //MidCspSubarrayBase.standardlibray_import

# tango imports
import tango
from tango import DebugIt
from tango.server import run
from tango.server import Device, DeviceMeta
from tango.server import attribute, command
from tango.server import device_property
from tango import AttrQuality, EventType, DevState
from tango import AttrWriteType, DeviceProxy
# Additional import
# PROTECTED REGION ID(MidCspMaster.additional_import) ENABLED START #
from skabase.SKAMaster  import SKAMaster
from skabase.auxiliary import utils
# import CSP.LMC Common package
from csp_lmc_common.utils.cspcommons import HealthState, AdminMode
from csp_lmc_common.CspSubarray import CspSubarray
# PROTECTED REGION END #    //  MidCspSubarrayBase.additionnal_import

__all__ = ["MidCspSubarrayBase", "main"]


class MidCspSubarrayBase(with_metaclass(DeviceMeta, CspSubarray)):
    """
    The base class for MID CspSubarray.
    Fuctionality to monitor assigned CSP.LMC Capabilities,
    as well as inherent Capabilities, are implemented in 
    separate TANGO Devices.

    **Properties:**

    - Class Property
    
    - Device Property
    """

    # PROTECTED REGION ID(MidCspSubarrayBase.class_variable) ENABLED START #
    # PROTECTED REGION END #    //  MidCspSubarrayBase.class_variable
    # PROTECTED REGION ID(MidCspSubarrayBase.private_methods) ENABLED START #
    def __monitor_add_receptors(self, receptor_list, args_dict= None):
        cmd_name = 'addreceptors'
        device = self.CbfSubarray
        self._num_dev_completed_task[cmd_name] = 0
        self._list_dev_completed_task[cmd_name] = []
        self._cmd_progress[cmd_name] = 0
        self._cmd_duration_measured[cmd_name] = 0
        # sub-component command execution measured time        
        self._sc_subarray_cmd_progress[device][cmd_name] = 0
        self._alarm_message[cmd_name] = ''
        device_proxy = self._sc_subarray_proxies[device]

        while True:
            # read the list of receptor IDs assigned to the CbfSubarray
            try:
                # Note: with numpy support in PyTango, receptors is a
                # numpy array
                receptors = device_proxy.receptors
            except tango.DevFailed as tango_err:
                self.logger.warn("__monitor_add_receptors:",tango_err.args[0].desc)
            if len(receptors):
                # get the ids in receptor_list that are also in receptor
                receptors_assigned = [value for value in receptor_list if value in receptors]
                self._num_dev_completed_task[cmd_name] = len(receptors_assigned)
                if len(receptors_assigned) == len(receptor_list):
                    self.logger.info("All required receptors asigned!!")
                    self._sc_subarray_cmd_progress[device][cmd_name] = 100
                    # calculate the real execution time for the command
                    self._cmd_duration_measured[cmd_name] = (time.time() - self._sc_subarray_cmd_starting_time[device])
                    break      
            # check if sub-element command ended throwing an exception: in this case the
            # 'cmd_ended_cb' callback is invoked.
            if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.FAILED:
                self._alarm_raised = True
                break
            elapsed_time = time.time() - self._sc_subarray_cmd_starting_time[device]
            if (elapsed_time > self._sc_subarray_cmd_duration_expected[device][cmd_name] or
                self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.TIMEOUT):
                msg = ("Timeout executing {} command  on device {}".format(cmd_name, device))
                self.logger.warn(msg)
                self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.TIMEOUT
                break
            # update the progress counter inside the loop taking into account the number of devices
            # executing the command
            self._cmd_progress[cmd_name] = self._sc_subarray_cmd_progress[device][cmd_name]
            time.sleep(1)    
        # end of the while loop
        
        self._last_executed_command = cmd_name
         # update the progress counter at the end of the loop 
        self._cmd_progress[cmd_name] = self._sc_subarray_cmd_progress[device][cmd_name]
        # check for error conditions
        if self._sc_subarray_state[self.CbfSubarray] != tango.DevState.ON:
            msg= ("AddReceptors: device {} is in {}"
                  " State".format(device,self._sc_subarray_state[self.CbfSubarray]))
            self.logger.warn(msg)
        if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.TIMEOUT:
            self._timeout_expired = True
        # reset the command execution state
        self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.IDLE
        self._cmd_execution_state[cmd_name] = CmdExecState.IDLE
     # PROTECTED REGION END #    //  MidCspSubarrayBase.private_methods
    
    def __monitor_remove_receptors(self, receptor_list, args_dict= None):
        cmd_name = 'removereceptors'
        device = self.CbfSubarray
        self._num_dev_completed_task[cmd_name] = 0
        self._list_dev_completed_task[cmd_name] = []
        self._cmd_progress[cmd_name] = 0
        self._cmd_duration_measured[cmd_name] = 0
        # sub-component command execution measured time        
        self._sc_subarray_cmd_progress[device][cmd_name] = 0
        self._alarm_message[cmd_name] = ''
        device_proxy = self._sc_subarray_proxies[device]
        while True:
            # read the list of receptor IDs assigned to the CbfSubarray
            try:
                receptors = device_proxy.receptors
            except AttributeError as attr_err:
                self.logger.warn("RemoveReceptors:",str(attr_err))
            if len(receptors):
                # get the ids in receptor_list that are no more present in receptors
                receptors_removed = [value for value in receptor_list if value not in receptors]
                self._num_dev_completed_task[cmd_name] = len(receptors) - len(receptor_list)
                if len(receptors_removed) == len(receptor_list):
                    self._sc_subarray_cmd_progress[device][cmd_name] = 100
                    # calculate the real execution time for the command
                    self._cmd_duration_measured[cmd_name] = (time.time() - self._sc_subarray_cmd_starting_time[device])
                    self.logger.info("Receptors {} have been successfully removed".format(receptor_list))
                    break
            # check if sub-element command ended throwing an exception: in this case the
            # 'cmd_ended_cb' callback is invoked.
            if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.FAILED:
                break
            elapsed_time = time.time() - self._sc_subarray_cmd_starting_time[device]
            if (elapsed_time > self._sc_subarray_cmd_duration_expected[device][cmd_name] or
                self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.TIMEOUT):
                msg = ("Timeout executing {} command  on device {}".format(cmd_name, device))
                self.logger.warn(msg)
                self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.TIMEOUT
                break
            # update the progress counter inside the loop taking into account the number of devices
            # executing the command
            self._cmd_progress[cmd_name] = self._sc_subarray_cmd_progress[device][cmd_name]
            time.sleep(1)    
        # end of the while loop
        self._last_executed_command = cmd_name
        # update the progress counter at the end of the loop 
        self._cmd_progress[cmd_name] = self._sc_subarray_cmd_progress[device][cmd_name]
        # check for error conditions
        if self._sc_subarray_state[self.CbfSubarray] not in [tango.DevState.OFF, 
                                                             tango.DevState.ON]:
            self._alarm_message[cmd_name] += ("Device {} is in {} State".format(device,
                                                                               self._sc_subarray_state[self.CbfSubarray]))
            self.logger.warn(self._alarm_message[device]) 
            self._alarm_raised = True
        if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.TIMEOUT:
            self._timeout_expired = True
        # reset the command exeuction state      
        self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.IDLE
        self._cmd_execution_state[cmd_name] = CmdExecState.IDLE
    
    def _validate_scan_configuration(self, json_config):
        """
        Overwritten method.
        Validate the MID scan configuration file.
        Currently it only copies the received configuration because it does not
        still exist any "cbf" block inside the JSON script.
        :param json_config: the input JSON formatted string with the configuration
                            for a scan
        """
        self.logger.debug("Validate scan configuration for MID CSP")
        #self._sc_subarray_obs_mode[self.CbfSubarray] = ObsMode.IMAGING
        #self._sc_scan_configuration[self.CbfSubarray] = json_config["cbf"]
        self._sc_subarray_scan_configuration[self.CbfSubarray] = json_config
        self._sc_subarray_assigned_fqdn.append(self.CbfSubarray)
             
    # PROTECTED REGION END #    //  MidCspSubarrayBase.private_methods
    # ----------------
    # Class Properties
    # ----------------


    # -----------------
    # Device Properties
    # -----------------
    # ----------
    # Attributes
    # ----------
    assignedFsp = attribute(
        dtype=('DevUShort',),
        max_dim_x=27,
        label="List of assigned FSPs",
        doc="List of assigned FSPs.",
    )

    assignedVcc = attribute(
        dtype=('DevUShort',),
        max_dim_x=197,
        label="List of assigned VCCs",
        doc="List of assigned VCCs.",
    )

    cbfOutputLink = attribute(name="cbfOutputLink",
        label="cbfOutputLink",
        forwarded=True
    )
    pssOutputLink = attribute(name="pssOutputLink",
        label="cbfOutputLink",
        forwarded=True
    )
    vlbiOutputLink = attribute(name="vlbiOutputLink",
        label="cbfOutputLink",
        forwarded=True
    )
    assignedVccState = attribute(name="assignedVccState",
        label="State of the assigned VCC",
        forwarded=True
    )
    assignedVccHealthState = attribute(name="assignedVccHealthState",
        label="HealthState of the assigned VCC",
        forwarded=True
    )
    assignedFspState = attribute(name="assignedFspState",
        label="State of the assigned FSPs",
        forwarded=True
    )
    assignedFspHealthState = attribute(name="assignedFspHealthState",
        label="HealthState of the assigned FSPs.",
        forwarded=True
    )
    assignedReceptors = attribute(name="assignedReceptors",
        label="The list of assigned receptor IDs.",
        forwarded=True
    )
    assignedVccObsState = attribute(name="assignedVccObsState",
        label="ObsState of the assigned VCCs",
        forwarded=True
    )
    assignedFspObsState = attribute(name="assignedFspObsState",
        label="ObsState of the assigned FSPs.",
        forwarded=True
    )
    assignedVccAdminMode = attribute(name="assignedVccAdminMode",
        label="AdminMode of the assigned VCCs.",
        forwarded=True
    )
    assignedFspAdminMode = attribute(name="assignedFspAdminMode",
        label="AdminMode of the assigned FSPs.",
        forwarded=True
    )
    # ---------------
    # General methods
    # ---------------

    def init_device(self):
        """Initialises the attributes and properties of the MidCspSubarrayBase."""
        CspSubarray.init_device(self)
        # PROTECTED REGION ID(MidCspSubarrayBase.init_device) ENABLED START #
        self._assigned_vcc = []              
        self._assigned_fsp = []
        self._receptor_to_vcc_map = {}
        self._receptor_id_list = []
        
         
        # PROTECTED REGION END #    //  MidCspSubarrayBase.init_device

    def always_executed_hook(self):
        """Method always executed before any TANGO command is executed."""
        # PROTECTED REGION ID(MidCspSubarrayBase.always_executed_hook) ENABLED START #
        # PROTECTED REGION END #    //  MidCspSubarrayBase.always_executed_hook

    def delete_device(self):
        """Hook to delete resources allocated in init_device.

        This method allows for any memory or other resources allocated in the
        init_device method to be released.  This method is called by the device
        destructor and by the device Init command.
        """
        # PROTECTED REGION ID(MidCspSubarrayBase.delete_device) ENABLED START #
        # PROTECTED REGION END #    //  MidCspSubarrayBase.delete_device
    # ------------------
    # Attributes methods
    # ------------------

    def read_assignedFsp(self):
        # PROTECTED REGION ID(MidCspSubarrayBase.assignedFsp_read) ENABLED START #
        """Return the assignedFsp attribute."""
        return (0,)
        # PROTECTED REGION END #    //  MidCspSubarrayBase.assignedFsp_read

    def read_assignedVcc(self):
        # PROTECTED REGION ID(MidCspSubarrayBase.assignedVcc_read) ENABLED START #
        """
        *Attribute method*

        :returns:
            The list of VCC IDs assigned to the subarray.

            *Type*: array of DevUShort.
        """
        # PROTECTED REGION ID(CspSubarray.vcc_read) ENABLED START #
        self._assigned_vcc.clear()
        # get the map VCC-receptor from CspMaster
        if not self._receptor_to_vcc_map:
            try:
                csp_proxy = tango.DeviceProxy(self.CspMaster)
                csp_proxy.ping()
                cbf_master_addr = csp_proxy.cbfMasterAddress
                cbf_master_proxy = tango.DeviceProxy(cbf_master_addr)
                cbf_master_proxy.ping()
                # build the list of receptor ids
                receptor_to_vcc = cbf_master_proxy.receptorToVcc
                self._receptor_to_vcc_map = dict([int(ID) for ID in pair.split(":")]
                                            for pair in receptor_to_vcc)
                # build the list of the installed receptors
                self._receptor_id_list = list(self._receptor_to_vcc_map.keys())
            except  tango.DevFailed as tango_err:
                self.logger.warn(tango_err.args[0].desc)
        try:
            assigned_receptors = self._sc_subarray_proxies[self.CbfSubarray].receptors
            # NOTE: if receptors attribute is empty, assigned_receptors is an empty numpy array
            # and it will just be skipped by the for loop
            for receptor_id in assigned_receptors:
                vcc_id = self._receptor_to_vcc_map[receptor_id]
                self._assigned_vcc.append(vcc_id)
        except KeyError as key_err:
            msg = "No {} found".format(key_err)
            tango.Except.throw_exception("Read attribute failure",
                                         msg,
                                         "read_vcc",
                                         tango.ErrSeverity.ERR)
        except tango.DevFailed as df:
            tango.Except.throw_exception("Read attribute failure",
                                         df.args[0].desc,
                                         "read_vcc",
                                         tango.ErrSeverity.ERR)
        return self._assigned_vcc
        # PROTECTED REGION END #    //  MidCspSubarrayBase.assignedVcc_read


    # --------
    # Commands
    # --------
    @AdminModeCheck('AddReceptors')
    @ObsStateCheck('addresources')
    def is_AddReceptors_allowed(self):
        return self._is_subarray_composition_allowed()
        
    @command(
        dtype_in='DevVarUShortArray',
        doc_in="List of the receptor IDs to add to the subarray.",
    )
    @DebugIt()
    #@SubarrayRejectCmd(['RemoveReceptors', 'Configure'])
    def AddReceptors(self, argin):
        # PROTECTED REGION ID(MidCspSubarrayBase.AddReceptors) ENABLED START #
        """
        *Class method*

        Add the specified receptor IDs to the subarray.

        The command can be executed only if the CspSubarray *ObsState* is *IDLE*.

        :param argin: the list of receptor IDs
        :type: array of DevUShort
        :return:
            None
        :raises:
            tango.DevFailed: if the CbfSubarray is not available or if an exception\
            is caught during command execution.
        Note:
            Still to implement the check on AdminMode values: the command can be processed \
            only when the CspSubarray is *ONLINE* or *MAINTENANCE*
        """
        # PROTECTED REGION ID(CspSubarray.AddReceptors) ENABLED START #
        # Each vcc_id map to a vcc_fqdn inside CbfMaster, for example:
        # vcc_id = 1 -> mid_csp_cbf/vcc/vcc_001
        # vcc_id = 2 -> mid_csp_cbf/vcc/vcc_002
        # .....
        # vcc_id = 17 -> mid_csp_cbf/vcc/vcc_017
        # vcc_id and receptor_id is not the same. The map between vcc_id and receptor_id
        # is built by CbfMaster and exported as attribute.
        # The max number of VCC allocated is defined by the VCC property of the CBF Master.

        # the list of available receptor IDs. This number is mantained by the CspMaster
        # and reported on request.
        available_receptors = []
        # the list of receptor to assign to the subarray
        receptor_to_assign = []
        try:
            # access to CspMaster to get information about the list of available receptors
            # and the receptors affiliation to subarrays.
            csp_master_proxy = tango.DeviceProxy(self.CspMaster)
            csp_master_proxy.ping()
            available_receptors = csp_master_proxy.unassignedReceptorIDs
            if not self._receptor_to_vcc_map:
                cbf_master_addr = csp_master_proxy.cbfMasterAddress
                cbf_master_proxy = tango.DeviceProxy(cbf_master_addr)
                cbf_master_proxy.ping()
                # build the list of receptor ids
                receptor_to_vcc = cbf_master_proxy.receptorToVcc
                self._receptor_to_vcc_map = dict([int(ID) for ID in pair.split(":")]
                                            for pair in receptor_to_vcc)
                # build the list of the installed receptors
                self._receptor_id_list = list(self._receptor_to_vcc_map.keys())
            print("available_receptors:", available_receptors)
            print("available_receptors:", type(available_receptors))
            if not any(available_receptors):
                log_msg = "No available receptor to add to subarray {}".format(self.SubID)
                self.dev_logging(log_msg, tango.LogLevel.LOG_WARN)
                return
            receptor_membership = csp_master_proxy.receptorMembership
            print("receptor_membership:", receptor_membership)
        except tango.DevFailed as df:
            msg = "Failure in getting receptors information:" + str(df.args[0].reason)
            tango.Except.throw_exception("Command failed", msg,
                                         "AddReceptors", tango.ErrSeverity.ERR)
        except AttributeError as attr_err:
            msg = "Failure in reading {}: {}".format(str(attr_err.args[0]), attr_err.__doc__)
            tango.Except.throw_exception("Command failed", msg,
                                         "AddReceptors", tango.ErrSeverity.ERR)
        for receptorId in argin:
            print("receptorId:", receptorId)
            print("self._receptor_id_list:", self._receptor_id_list)
            # check if the specified receptor id is a valid number (that is, belongs to the list
            # of provided receptors)
            if receptorId in self._receptor_id_list:
                # check if the receptor id is one of the available receptor Ids
                if receptorId in available_receptors:
                    receptor_to_assign.append(receptorId)
                else:
                    # retrieve the subarray owner
                    sub_id = receptor_membership[receptorId - 1]
                    log_msg = "Receptor {} already assigned to subarray {}".format(str(receptorId),
                                                                                   str(sub_id))
                    self.logger.info(log_msg)
            else:
                log_msg = "Invalid receptor id: {}".format(str(receptorId))
                self.logger.warn(log_msg)

        # check if the list of receptors to assign is empty
        if not receptor_to_assign:
            log_msg = "The required receptors are already assigned to a subarray"
            self.logger.info(log_msg)
            return
        # check if the CspSubarray is already connected to the CbfSubarray
        proxy = 0
        if self._is_sc_subarray_running(self.CbfSubarray):
            self._cmd_execution_state['addreceptors'] = CmdExecState.RUNNING
            proxy = self._sc_subarray_proxies[self.CbfSubarray]
            # remove possible receptor repetition
            tmp = set(receptor_to_assign)
            receptor_to_assign = list(tmp)
            # forward the command to the CbfSubarray
            proxy.command_inout_asynch("AddReceptors", receptor_to_assign, self._cmd_ended_cb)
            self._sc_subarray_cmd_starting_time['addreceptors'] = time.time()
            self._command_thread['addreceptors'] = threading.Thread(target=self.__monitor_add_receptors,
                                                           name="Thread-AddReceptors",
                                                           args=(receptor_to_assign,))
            self._sc_subarray_cmd_starting_time['addreceptors'] = time.time()
            self._command_thread['addreceptors'].start()
        else:
            log_msg = "Device {} is not running!".format(str(self._cbf_subarray_fqdn))
            self.logger.error(log_msg)
            tango.Except.throw_exception("Command failed",
                                         log_msg,
                                         "AddReceptors",
                                         tango.ErrSeverity.ERR)
        # PROTECTED REGION END #    //  MidCspSubarrayBase.AddReceptors
    @AdminModeCheck('RemoveReceptors')
    @ObsStateCheck('removeresources')
    def is_RemoveReceptors_allowed(self):
        
        return self._is_subarray_composition_allowed()
    
    @command(
        dtype_in='DevVarUShortArray',
        doc_in="The list with the receptor IDs to remove",
    )
    @DebugIt()
    @SubarrayRejectCmd(['AddReceptors', 'Configure'])
    def RemoveReceptors(self, argin):
        # PROTECTED REGION ID(MidCspSubarrayBase.RemoveReceptors) ENABLED START #
        """
        Remove the receptor IDs from the subarray.

        :param argin: The list of the receptor IDs to remove from the subarray.
            Type: array of DevUShort
        :returns:
            None
        :raises:
            tango.DevFailed: raised if the subarray *obState* attribute is not IDLE, or \
                    when an exception is caught during command execution.
        """
        # PROTECTED REGION ID(CspSubarray.RemoveReceptors) ENABLED START #  
        # check if the CspSubarray is already connected to the CbfSubarray
        if self._is_sc_subarray_running(self.CbfSubarray):
            try:
                
                proxy = self._sc_subarray_proxies[self.CbfSubarray]
                # read from CbfSubarray the list of assigned receptors
                # failure in reading attribute raises an AttributeError (not tangoDevFailed!!)
                receptors = proxy.receptors
                #!!!!!!!!!!!!!!!!!
                # 2019-09-20:  New images for TANGO and PyTango images has been released. PyTango
                # is now compiled with th numpy support. In this case the proxy.receptors call
                # does no more return an empty tuple but an empty numpy array.
                # Checks on receptors content need to be changed (see below)
                # NB: the receptors attribute implemented by the CbfSubarray is declared as RW.
                # In this case the read method returns an empty numpy array ([]) whose length is 0.
                #!!!!!!!!!!!!!!!!!

                # check if the list of assigned receptors is empty
                if len(receptors):
                    receptors_to_remove = []
                    # check if the receptors to remove belong to the subarray
                    for receptor_id in argin:
                        if receptor_id in receptors:
                            receptors_to_remove.append(receptor_id)
                    if any(receptors_to_remove):
                        #TODO_ add reading of CbfSubarray removeReceptorsCmdDurationExpected
                        # attribute, if implemented, otherwise use the default value
                        
                        # subscribes attributes to track progress and timeout: if these
                        # attributes are not implemented at CbfSubarray level, the warning
                        # is only logged because is not a fatal error.
                        for attr in ['removeReceptorsCmdProgress', 'cmdTimeoutExpired']:
                            try:
                                if self._sc_subarray_event_id[self.CbfSubarray][attr.lower()] == 0:
                                    evt_id = proxy.subscribe_event(attr, tango.EventType.CHANGE_EVENT,
                                                        self._attributes_change_evt_cb, stateless=False)
                                    self._sc_subarray_event_id[self.CbfSubarray][attr.lower()] = evt_id
                            except tango.DevFailed as tango_err:
                                self.logger.info(tango_err.args[0].desc)
                        # forward the command to CbfSubarray
                        proxy.command_inout_asynch("RemoveReceptors", receptors_to_remove, self._cmd_ended_cb)
                        self._cmd_execution_state['removereceptors'] = CmdExecState.RUNNING
                        self._sc_subarray_cmd_starting_time['removereceptors'] = time.time()
                        # Note: rembember to put the comma in args=(receptors_to_remove,) otherwise
                        # the list is received as a numpy array!
                        self._command_thread['removereceptors'] = threading.Thread(target=self.__monitor_remove_receptors,
                                                                name="Thread-RemoveReceptors",
                                                                args=(receptors_to_remove,))
                        
                        self._command_thread['removereceptors'].start()
                self.logger.info("No receptor to remove from subarray {}".format(self.get_name()))
                self._cmd_execution_state['removereceptors'] = CmdExecState.IDLE
                return
            except tango.DevFailed as tango_err:
                tango.Except.throw_exception("Command failed",
                                             tango_err.args[0].desc,
                                             "RemoveReceptors",
                                             tango.ErrSeverity.ERR)
            except AttributeError as attr_err:
                print("ATTRERROR")
                log_msg = "RemoveReceptors:" + str(attr_err)
                self.logger.error(log_msg)
                tango.Except.throw_exception("Command failed",
                                                str(attr_err),
                                                "RemoveReceptors",
                                                tango.ErrSeverity.ERR)
        else:
            log_msg = "Subarray {} not registered!".format(str(self.CbfSubarray))
            self.logger.error(log_msg)
            tango.Except.throw_exception("Command failed",
                                         log_msg,
                                         "RemoveReceptors",
                                         tango.ErrSeverity.ERR)
       
        # PROTECTED REGION END #    //  MidCspSubarrayBase.RemoveReceptors

    @command(
    )
    @DebugIt()
    def RemoveAllReceptors(self):
        # PROTECTED REGION ID(MidCspSubarrayBase.RemoveAllReceptors) ENABLED START #
        """
        *Class method.*

        Remove all the assigned receptors from the subarray.
        Returns:
            None
        Raises:
            tango.DevFailed: raised if the subarray *obState* attribute is not IDLE or READY, or \
            when an exception is caught during command execution.
        """
        # PROTECTED REGION ID(CspSubarray.RemoveAllReceptors) ENABLED START #
        if self._is_sc_subarray_running(self.CbfSubarray):
            try:
                proxy = self._sc_subarray_proxies[self.CbfSubarray]
                # check if the list of assigned receptors is empty
                receptors = proxy.receptors
                if len(receptors):
                    self.RemoveReceptors(receptors[:])
            except tango.DevFailed as df:
                log_msg = ("RemoveAllReceptors failure. Reason: {} "
                           "Desc: {}".format(df.args[0].reason,
                                             df.args[0].desc))
                self.dev_logging(log_msg, tango.LogLevel.LOG_ERROR)
                tango.Except.re_throw_exception(df, "Command failed",
                                                "CspSubarray RemoveAllReceptors command failed",
                                                "Command()",
                                                tango.ErrSeverity.ERR)
        else:
            log_msg = "Subarray {} not registered!".format(str(self._cbf_subarray_fqdn))
            self.dev_logging(log_msg, tango.LogLevel.LOG_ERROR)
            tango.Except.throw_exception("Command failed",
                                         log_msg,
                                         "RemoveAllReceptors",
                                         tango.ErrSeverity.ERR)
        
        # PROTECTED REGION END #    //  MidCspSubarrayBase.RemoveAllReceptors

# ----------
# Run server
# ----------


def main(args=None, **kwargs):
    # PROTECTED REGION ID(MidCspSubarrayBase.main) ENABLED START #
    return run((MidCspSubarrayBase,), args=args, **kwargs)
    # PROTECTED REGION END #    //  MidCspSubarrayBase.main

if __name__ == '__main__':
    main()
