# -*- coding: utf-8 -*-
#
# This file is part of the CspSubarray project
#
# INAF - SKA Telescope
#
# Distributed under the terms of the GPL license.
# See LICENSE.txt for more info.

""" CSP.LMC Common CspSubarray

CSP subarray functionality is modeled via a TANGCSP.LMC Common Class for the CSPSubarray TANGO Device.
"""
# PROTECTED REGION ID (CspSubarray.standardlibray_import) ENABLED START #
# Python standard library
from __future__ import absolute_import
import sys
import os
from future.utils import with_metaclass
from collections import defaultdict
from enum import IntEnum, unique
import threading
import time
import json
# PROTECTED REGION END# //CspSubarray.standardlibray_import

# tango imports
import tango
from tango import DebugIt
from tango.server import run
from tango.server import Device, DeviceMeta
from tango.server import attribute, command
from tango.server import class_property, device_property
from tango import AttrQuality, DispLevel, DevState
from tango import AttrWriteType, PipeWriteType

# Additional import
# PROTECTED REGION ID(CspSubarray.additionnal_import) ENABLED START #
from skabase.SKASubarray  import SKASubarray
from skabase.auxiliary import utils
# PROTECTED REGION END #    //  CspSubarray.additionnal_import

# PROTECTED REGION ID (CspSubarray.add_path) ENABLED START #
# add the path to import global_enum package.

from .utils.decorators import AdminModeCheck, StateAndObsStateCheck
from .utils.cspcommons import HealthState, AdminMode, ObsState, ObsMode, CmdExecState
from . import release
# PROTECTED REGION END# //CspSubarray.add_path

__all__ = ["CspSubarray", "main"]


class CspSubarray(with_metaclass(DeviceMeta,SKASubarray)):
    """
    CSP subarray functionality is modeled via a TANGCSP.LMC Common Class for the CSPSubarray TANGO Device.

    **Properties:**

    - Class Property
        PSTBeams
            - PST sub-element PSTBeams TANGO devices FQDNs
            - Type:'DevVarStringArray'
    
    - Device Property
    
        CspMaster
            - The TANGO address of the CspMaster.
            - Type:'DevString'
    
        CbfSubarray
            - CBF sub-element sub-array TANGO device FQDN
            - Type:'DevString'
    
        PssSubarray
            - PST sub-element sub-array TANGO device FQDN.
            - Type:'DevString'
    
        SubarrayProcModeCorrelation
            - CSP Subarray *Correlation Inherent Capability*\nTANGO device FQDN
            - Type:'DevString'
    
        SubarrayProcModePss
            - CSP Subarray *Pss Inherent Capability*\nTANGO device FQDN
            - Type:'DevString'
    
        SubarrayProcModePst
            - CSP Subarray *PST Inherent Capability*\nTANGO device FQDN
            - Type:'DevString'
    
        SubarrayProcModeVlbi
            - CSP Subarray *VLBI Inherent Capability*\nTANGO device FQDN
            - Type:'DevString'
    
    """
    # PROTECTED REGION ID(CspSubarray.class_variable) ENABLED START #
    # PROTECTED REGION END #    //  CspSubarray.class_variable
    # !! NOTE !!: 
    # In methods and attributes of the class:
    # 'sc' prefix stands for 'sub-component'
    # 'cb' suffix stands for 'callback'
    #----------------
    # Event Callback functions
    # ---------------
    def _sc_scm_change_event_cb(self, evt):
        """
        Class protected callback function.
        Retrieve the values of the sub-array sub-component SCM attributes subscribed
        at device connection.
        Sub-array sub-components are:
        - the CBF sub-array (assigned at initialization!)
        - the PSS sub-array if SearchBeams are assigned to the sub-array
        - the PSTBeams if TimingBeams are assigned to the sub-array
        These values are used to report the whole Csp Subarray State and healthState.

        :param evt: The event data

        :return: None
        """
        dev_name = evt.device.dev_name()
        if not evt.err:
            try:
                if dev_name in self._sc_subarray_fqdn:
                    if evt.attr_value.name.lower() == "state":
                        self._sc_subarray_state[dev_name] = evt.attr_value.value
                    elif evt.attr_value.name.lower() == "healthstate":
                        self._sc_subarray_health_state[dev_name] = evt.attr_value.value
                    elif evt.attr_value.name.lower() == "adminmode":
                        self.logger.debug("device: {} adminMode value {}".format(dev_name,evt.attr_value.value))
                        self._sc_subarray_admin_mode[dev_name] = evt.attr_value.value
                    elif evt.attr_value.name.lower() == "obsstate":
                        self.logger.debug("device: {} obsState value {}".format(dev_name,evt.attr_value.value ))
                        self._sc_subarray_obs_state[dev_name] = evt.attr_value.value
                    elif evt.attr_value.name.lower() == "obsmode":
                        self.logger.debug("device: {} obsMode value {}".format(dev_name, evt.attr_value.value ))
                        self._sc_subarray_obs_mode[dev_name] = evt.attr_value.value
                    else:
                        log_msg = ("Attribute {} not still "
                                   "handled".format(evt.attr_name))
                        self.logger.warn(log_msg)
                else:
                    log_msg = ("Unexpected change event for"
                               " attribute: {}".format(str(evt.attr_name)))
                    self.logger.warn(log_msg)
                    return

                log_msg = "New value for {} is {}".format(str(evt.attr_name),
                                                          str(evt.attr_value.value))
                self.logger.info(log_msg)
                # update CSP sub-array SCM
                if evt.attr_value.name.lower() in ["state", "healthstate", "adminmode"]:
                    self._update_subarray_state()
                if evt.attr_value.name.lower() == "obsstate":
                   self._update_subarray_obs_state()
            except tango.DevFailed as df:
                self.logger.error(str(df.args[0].desc))
            except Exception as except_occurred:
                self.logger.error(str(except_occurred))
        else:
            for item in evt.errors:
                # API_EventTimeout: if sub-element device not reachable it transitions
                # to UNKNOWN state.
                if item.reason == "API_EventTimeout":
                    # only if the device is ONLINE/MAINTENANCE, its State is set to 
                    # UNKNOWN when there is a timeout on connection, otherwise its
                    # State should be reported always as DISABLE
                    if self._sc_subarray_admin_mode[dev_name] in [AdminMode.ONLINE,
                                                         AdminMode.MAINTENANCE]:
                        self._sc_subarray_state[dev_name] = tango.DevState.UNKNOWN
                        self._sc_subarray_health_state[dev_name] = HealthState.UNKNOWN
                        # TODO how report obsState? 
                        # adminMode can't be change otherwise the State and healthState
                        # are note updated
                    # update the State and healthState of the CSP sub-array
                    self._update_subarray_state()
                log_msg = item.reason + ": on attribute " + str(evt.attr_name)
                self.logger.warn(log_msg)
    
    def _attributes_change_evt_cb(self, evt):
        """
        *Class callback function.*
        Retrieve the value of the sub-element xxxCommandProgress attribute
        subscribed for change event when a long-running command is issued
        on the sub-element device.

        :param evt: The event data

        :return: None
        """
        dev_name = evt.device.dev_name()
        if not evt.err:
            try:
                if "commandprogress" == evt.attr_value.name.lower()[-15:]:
                    if dev_name in self._sc_subarray_fqdn:
                        cmd_name = evt.attr_value.name[:-15]
                        self._sc_subarray_cmd_progress[dev_name][cmd_name] = evt.attr_value.value
                elif "cmdtimeoutexpired" == evt.attr_value.name.lower()[-17:]:
                    if dev_name in self._sc_subarray_fqdn:
                        cmd_name = evt.attr_value.name[:-17]
                        self._sc_subarray__timeout_expired[dev_name][cmd] = True
                else:
                    log_msg = ("Unexpected change event for"
                               " attribute: {}".format(str(evt.attr_name)))
                    self.logger.warn(log_msg)
                    return

                log_msg = "New value for {} is {}".format(str(evt.attr_name),
                                                          str(evt.attr_value.value))
                self.logger.info(log_msg)
            except tango.DevFailed as df:
                self.logger.error(str(df.args[0].desc))
            except Exception as except_occurred:
                self.logger.error(str(except_occurred))
        else:
            for item in evt.errors:
                log_msg = item.reason + ": on attribute " + str(evt.attr_name)
                self.logger.warn(log_msg)
              
    def _cmd_ended_cb(self, evt):
        """
        Callback function immediately executed when the asynchronous invoked
        command returns.

        :param evt: a CmdDoneEvent object. This class is used to pass data
            to the callback method in asynchronous callback model for command
            execution.
        :type: CmdDoneEvent object
            It has the following members:
                - device     : (DeviceProxy) The DeviceProxy object on which the
                               call was executed.
                - cmd_name   : (str) The command name
                - argout_raw : (DeviceData) The command argout
                - argout     : The command argout
                - err        : (bool) A boolean flag set to true if the command
                               failed. False otherwise
                - errors     : (sequence<DevError>) The error stack
                - ext
        :return: none
        """
        # NOTE:if we try to access to evt.cmd_name or other paramters, sometime
        # the callback crashes withthis error:
        # terminate called after throwing an instance of 'boost::python::error_already_set'
        try:
            # Can happen evt empty??
            if evt:
                if not evt.err:
                    if self._sc_subarray_cmd_exec_state[evt.device.dev_name()][evt.cmd_name.lower()] == CmdExecState.RUNNING:
                        msg = "Device {} is processing command {}".format(evt.device,
                                                                      evt.cmd_name)
                    if self._sc_subarray_cmd_exec_state[evt.device.dev_name()][evt.cmd_name.lower()] == CmdExecState.IDLE:
                        msg = "Device {} ended command {} execution".format(evt.device, evt.cmd_name)
                    self.logger.info(msg)
                else:
                    msg = "Error!!Command {} ended on device {}.\n".format(evt.cmd_name,
                                                                           evt.device.dev_name())
                    msg += " Desc: {}".format(evt.errors[0].desc)
                    self.logger.info(msg)
                    self._sc_subarray_cmd_exec_state[evt.device.dev_name()][evt.cmd_name.lower()] = CmdExecState.FAILED
                    self._alarm_message[evt.cmd_name.lower()] += msg
                    # obsState and obsMode values take on the CbfSubarray's values via
                    # the subscribe/publish mechanism
            else:
                self.logger.error("cmd_ended callback: evt is empty!!")
        except tango.DevFailed as df:
            msg = ("CommandCallback cmd_ended failure - desc: {}"
                   " reason: {}".format(df.args[0].desc, df.args[0].reason))
            self.logger.error(msg)
        except Exception as ex:
            msg = "CommandCallBack cmd_ended general exception: {}".format(str(ex))
            self.logger.error(msg)
           
    # Class protected methods
    # ---------------
    
    def _update_subarray_state(self):
        """
        Class protected method.
        Retrieve the State attribute values of the CSP sub-elements and aggregate
        them to build up the CSP global state.

        :param: None

        :return: None
        """
        self._update_subarray_health_state()
        # CSP state reflects the status of CBF. Only if CBF is present
        # CSP can work. The state of PSS and PST sub-elements only contributes
        # to determine the CSP health state.
        self.set_state(self._sc_subarray_state[self.CbfSubarray])
        if self._sc_subarray_admin_mode[self.CbfSubarray] not in [AdminMode.ONLINE,
                                                                  AdminMode.MAINTENANCE]:
            self.set_state(tango.DevState.DISABLE)
        if self._admin_mode not in [AdminMode.ONLINE,
                                    AdminMode.MAINTENANCE]:
            self.set_state[tango.DevState.DISABLE]

    def _update_subarray_health_state(self):
        """
        Class protected method.
        Retrieve the healthState attribute of the CSP sub-elements and
        aggregate them to build up the CSP health state

        :param: None

        :return: None
        """
        # if the Subarray is OFF (no assigned resources) or DISABLE,
        # its health state is UNKNOWN.
        # Note: when the Subarray adminMode is set OFFLINE/RESERVED/NOTFITTED
        # all its allocated resources are released and its State moves to DISABLE.
        # 
        if self.get_state() in [tango.DevState.OFF, tango.DevState.DISABLE]:
            self._health_state = HealthState.UNKNOWN
            
        # The whole CspSubarray HealthState is OK if is ON and all its assigned sub-components
        # (CBF and PSS subarrays as well PST Beams) are OK.
        # - CbfSubarray ON (receptors/stations assigned)
        # - PssSubarray ON
                
        # default value to DEGRADED
        self._health_state = HealthState.DEGRADED
        
        # build the list of all the Csp Subarray sub-components ONLINE/MAINTENANCE
        admin_fqdn = [fqdn for fqdn, admin_value in self._sc_subarray_admin_mode.items()
                      if admin_value in [AdminMode.ONLINE, AdminMode.MAINTENANCE]]
        # build the list of sub-elements with State ON
        state_fqdn =  [fqdn for fqdn in admin_fqdn if self._sc_subarray_state[fqdn] == tango.DevState.ON]
        # build the list with the healthState of ONLINE/MAINTENANCE devices 
        health_list = [self._sc_subarray_health_state[fqdn] for fqdn in state_fqdn]
        
        if self.CbfSubarray in admin_fqdn:
            if all(value == HealthState.OK for value in health_list):
                self._health_state = HealthState.OK
            elif self._sc_subarray_health_state[self.CbfSubarray] in [HealthState.FAILED,
                                                             HealthState.UNKNOWN,
                                                             HealthState.DEGRADED]:
                self._health_state = self._sc_subarray_health_state[self.CbfSubarray]
        else:
            # if CBF Subarray is not ONLINE/MAINTENANCE ....
            self._health_state = self._sc_subarray_health_state[self.CbfSubarray]
        return
    
    def _update_subarray_obs_state(self):
        """
        Class protected method.
        Retrieve the State attribute values of the CSP sub-elements and aggregate
        them to build up the CSP global state.

        :param: None

        :return: None
        """
        # update the CspSubarray ObsMode only when no command is running on sub-array
        # sub-components
        exec_state_list = []
        for fqdn in self._sc_subarray_assigned_fqdn:
            device_exec_state_list = [value for value in self._sc_subarray_cmd_exec_state[fqdn].values()]
            exec_state_list.extend(device_exec_state_list)
            # when command are not running, the CbfSubarray osbState reflects the
            # CSP Subarray obsState.
            # If CbfSubarray obsState is READY and PssSubarray IDLE, the CspSubarray
            # onsState is READY (it can performs IMAGING!)
            if all(value == CmdExecState.IDLE for value in exec_state_list):
                self._obs_state = self._sc_subarray_obs_state[self.CbfSubarray]
                self.logger.debug("All sub-array sub-component are IDLE."
                                  "Subarray obsState:", self._obs_state)
                
    def _connect_to_subarray_subcomponent(self, fqdn):
        """
        Class method.
        Establish a *stateless* connection with a CSP Subarray sub-component
        device (CBF Subarray, PSS Subarray , PST Beams).
        Exceptions are logged.
        :param fqdn:
                the CspSubarray sub-component FQDN
        :return: None
        """
        
        # check if the device has already been addedd
        if fqdn in self._sc_subarray_fqdn:
            return
        # read the sub-componet adminMode (memorized) attribute from
        # the CSP.LMC TANGO DB. 
        attribute_properties = self._csp_tango_db.get_device_attribute_property(fqdn, 
                                                                             {'adminMode': ['__value']})
        self.logger.info("fqdn: {} attribute_properties: {}".format(fqdn, attribute_properties))
        try:
            admin_mode_memorized = attribute_properties['adminMode']['__value']
            self._sc_subarray_admin_mode[fqdn] = int(admin_mode_memorized[0])
        except KeyError:
            self.logger.warn("No key {} found".format(str(key_error)))    
        try:
            log_msg = "Trying connection to " + str(fqdn) + " device"
            self.logger.info(log_msg)
            device_proxy = tango.DeviceProxy(fqdn)
            # Note: The DeviceProxy is initialized even if the sub-component
            # device is not running (but defined into the TANGO DB! If not defined in the
            # TANGO DB a exception is throw). 
            # The connection with a sub-element is establish as soon as the corresponding
            # device starts. 
            # append the FQDN to the list and store the sub-element proxies
            self._sc_subarray_fqdn.append(fqdn)
            self._sc_subarray_proxies[fqdn] = device_proxy
            
            # subscription of SCM attributes (State, healthState and adminMode).
            # Note: subscription is performed also for devices not ONLINE or MAINTENANCE.
            # In this way the CspMaster is able to detect a change in the admin value.
               
            ev_id = device_proxy.subscribe_event("adminMode",
                                                 tango.EventType.CHANGE_EVENT,
                                                 self._sc_scm_change_event_cb,
                                                 stateless=True)
            self._sc_subarray_event_id[fqdn]['adminMode'] = ev_id
                
            ev_id = device_proxy.subscribe_event("State",
                                                 tango.EventType.CHANGE_EVENT,
                                                 self._sc_scm_change_event_cb,
                                                 stateless=True)
            self._sc_subarray_event_id[fqdn]['state'] = ev_id

            ev_id = device_proxy.subscribe_event("healthState",
                                                 tango.EventType.CHANGE_EVENT,
                                                 self._sc_scm_change_event_cb,
                                                 stateless=True)
            self._sc_subarray_event_id[fqdn]['healthState'] = ev_id
            
            ev_id = device_proxy.subscribe_event("obsState",
                                                 tango.EventType.CHANGE_EVENT,
                                                 self._sc_scm_change_event_cb,
                                                 stateless=True)
            self._sc_subarray_event_id[fqdn]['obsState'] = ev_id
                
            ev_id = device_proxy.subscribe_event("obsMode",
                                                 tango.EventType.CHANGE_EVENT,
                                                 self._sc_scm_change_event_cb,
                                                 stateless=True)
            self._sc_subarray_event_id[fqdn]['obsMode'] = ev_id
                
        except KeyError as key_err:
            log_msg = ("No key {} found".format(str(key_err)))
            self.logger.warn(log_msg)
        except tango.DevFailed as df:
            log_msg = ("Failure in connection to {}"
                       " device: {}".format(str(fqdn), str(df.args[0].desc)))
            self.logger.error(log_msg)

    def _is_sc_subarray_running (self, device_name):
        """
        *Class protected method.*

        Check if a sub-element is exported in the TANGO DB (i.e its TANGO 
        device server is running).
        If the device is not in the list of the connected sub-elements,
        a connection with the device is performed.

        :param: subelement_name : the FQDN of the sub-element
        :type: `DevString`
        :return: True if the connection with the subarray is established,
                False otherwise
        """
        try:
            proxy = self._sc_subarray_proxies[device_name]
            proxy.ping()
        except KeyError as key_err:
            # Raised when a mapping (dictionary) key is not found in the set
            # of existing keys.
            # no proxy registered for the subelement device
            msg = "Can't retrieve the information of key {}".format(key_err)
            self.logger.warn(msg)
            try: 
                proxy = tango.DeviceProxy(device_name)
                # execute a ping to detect if the device is actually running
                proxy.ping()
                self._sc_subarray_proxies[device_name] = proxy
            except tango.DevFailed as df:  
                return False  
        except tango.DevFailed as df:
            msg = "Failure reason: {} Desc: {}".format(str(df.args[0].reason), str(df.args[0].desc))
            self.logger.warn(msg)
            return False
        return True
    
    def _is_subarray_composition_allowed(self):
        if self.get_state() in [tango.DevState.ON, tango.DevState.OFF]:
            return True
        return False
    
    def _is_subarray_configuring_allowed(self):
        if self.get_state() == tango.DevState.ON:
            return True
        return False
    
    # ----------------
    # Class private methods
    # ----------------
    
    def __configure_scan(self, device_list, **args_dict):
        tango_cmd_name = 0
        # the sub-element device state after successful transition
        dev_successful_state = 0
        try:
            tango_cmd_name = args_dict['cmd_name']
            dev_successful_state = args_dict['obs_state']
        except KeyError as key_err:
            self.logger.warn("No key: {}".format(str(key_err)))
            return
        # tango_cmd_name: is the TANGO command name with the capital letter
        # In the dictionary keys, is generally used the command name in lower letters
        cmd_name = tango_cmd_name.lower()
        self.logger.info("cmd_name: {} dev_state: {}".format(cmd_name,
                          dev_successful_state))
        self._num_dev_completed_task[cmd_name] = 0
        self._list_dev_completed_task[cmd_name] = []
        self._cmd_progress[cmd_name] = 0
        self._cmd_duration_measured[cmd_name] = 0
        # sub-component command execution measured time
        sc_cmd_duration_measured = defaultdict(lambda:defaultdict(lambda:0))
        # loop on the devices and issue asynchrnously the ConfigureScan command
        for device in device_list:
            proxy = self._sc_subarray_proxies[device]
            sc_cmd_duration_measured[device][cmd_name] = 0
            self._sc_subarray_cmd_progress[device][cmd_name] = 0
            self._alarm_message[cmd_name] = ''
            # Note: CBF/PSS sub-array checks for the validity of its
            # configuration. Failure in configuration throws an exception that is
            # caught via the _cmd_ended_cb callback
            proxy.command_inout_asynch("ConfigureScan", self._sc_subarray_scan_configuration[device], self._cmd_ended_cb)
            self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.RUNNING
            self._obs_state = ObsState.CONFIGURING
            # register the starting time for the command
            self._sc_subarray_cmd_starting_time[device] = time.time() 
            self.logger.info("Device {} State {} expected value {}".format(device,
                                                                           self._sc_subarray_state[device], 
                                                                           dev_successful_state))
        command_progress = self._cmd_progress[cmd_name]
        # flag to signal when configuration ends on a sub-array sub-component
        device_done = defaultdict(lambda:False)
        # inside the end-less lop check the obsState of each sub-component
        while True:
            for device in device_list:
                print("__configure_scan obs_state:", self._sc_subarray_obs_state[device])
                if device_done[device] == True:
                    continue
                # if the sub-component execution flag is no more RUNNING, the command has
                # ended with or without success. Go to check next device state.
                if self._sc_subarray_obs_state[device] == dev_successful_state:
                    self.logger.info("Command {} ended with success on device {}.".format(cmd_name,
                                                                                              device))
                    # update the list and number of device that completed the task
                    self._num_dev_completed_task[cmd_name]  += 1
                    self._list_dev_completed_task[cmd_name].append(device)
                    # reset the value of the attribute reporting the execution state of
                    # the command
                    self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.IDLE
                    self._sc_subarray_cmd_progress[device][cmd_name] = 100
                    # calculate the real execution time for the command
                    self._cmd_duration_measured[cmd_name] += sc_cmd_duration_measured[device][cmd_name]
                    # command success: step to next device
                    device_done[device] = True
                    # check if sub-element command ended throwing an exception: in this case the
                    # 'cmd_ended_cb' callback is invoked.
                if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.FAILED:
                    # execution ended for this sub-element, skip to the next one
                    device_done[device] = True
                #if self._sc_subarray_obs_state[device] in [ObsState.ABORTED, ObsState.FAULT]:
                #    num_of_failed_device += 1
                #    continue
                # check for timeout event. A timeout event can be detected in two ways:
                # 1- the sub-element implements the 'onTimeoutExpired' attribute configured
                #    for change event
                # 2- the CspMaster periodically checks the time elapsed from the start
                #    of the command: if the value is greater than the sub-element expected time
                #    for command execution, the sub-element command execution state is set
                #    to TIMEOUT
                # Note: the second check, can be useful if the timeout event is not received
                # (for example for a temporary connection timeout)
                elapsed_time = time.time() - self._sc_subarray_cmd_starting_time[device]
                if (elapsed_time > self._sc_subarray_cmd_duration_expected[device][cmd_name] or
                    self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.TIMEOUT):
                    msg = ("Timeout executing {} command  on device {}".format(cmd_name, device))
                    self.logger.warn(msg)
                    self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.TIMEOUT
                    device_done[device] = True 
                    self.logger.info("elapsed_time:{} device {}".format(elapsed_time, device))
                # update the progress counter inside the loop taking into account the number of devices
                # executing the command
                self._cmd_progress[cmd_name] = command_progress + self._sc_subarray_cmd_progress[device][cmd_name]/len(device_list)
                print("__configure_scan:", self._sc_subarray_cmd_exec_state[device][cmd_name])
            if all(value == True for value in device_done.values()):
                self.logger.info("All devices have been handled!")
                if (all(self._sc_subarray_obs_state[device] == ObsState.READY for device in device_list)):
                    self._obs_state = ObsState.READY
                    # end of the command: the command has been issued on all the sub-element devices
                    # reset the execution flag for the CSP
                break
            time.sleep(1)
        # end of the while loop
        # check for timeout/alarm conditions on each sub-component
        for device in device_list:
            if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.TIMEOUT:
                self._timeout_expired = True
            if self._sc_subarray_cmd_exec_state[device][cmd_name] == CmdExecState.FAILED:
                self._alarm_raised = True
            # reset sub-component execution flag
            self._sc_subarray_cmd_exec_state[device][cmd_name] = CmdExecState.IDLE
            # update the progress counter at the end of the loop 
            self._cmd_progress[cmd_name] = command_progress + self._sc_subarray_cmd_progress[device][cmd_name]/len(device_list)
        if all(self._sc_subarray_obs_state[fqdn] == dev_successful_state for fqdn in device_list):
            self._obs_state = dev_successful_state
        self._last_executed_command = cmd_name
        # reset the CSP Subarray command execution flag
        self._cmd_execution_state[cmd_name] = CmdExecState.IDLE

    # ----------------
    # Class Properties
    # ----------------

    PSTBeams = class_property(
        dtype='DevVarStringArray',
    )

    # -----------------
    # Device Properties
    # -----------------
    
    CspMaster = device_property(
        dtype='DevString', 
    )

    CbfSubarray = device_property(
        dtype='DevString',
    )

    PssSubarray = device_property(
        dtype='DevString',
    )

    SubarrayProcModeCorrelation = device_property(
        dtype='DevString',
    )

    SubarrayProcModePss = device_property(
        dtype='DevString',
    )

    SubarrayProcModePst = device_property(
        dtype='DevString',
    )

    SubarrayProcModeVlbi = device_property(
        dtype='DevString',
    )

    # ----------
    # Attributes
    # ----------
    
    scanID = attribute(
        dtype='DevULong64',
        access=AttrWriteType.READ_WRITE,
    )

    procModeCorrelationAddr = attribute(
        dtype='DevString',
        label="Correlation Inherent Capability Address",
        doc="The CSP sub-array Correlation Inherent Capability FQDN.",
    )

    procModePssAddr = attribute(
        dtype='DevString',
        label="PSS Inherent Capability address",
        doc="The CSP sub-array PSS Inherent Capability FQDN.",
    )

    procModePstAddr = attribute(
        dtype='DevString',
        label="PST Inherent Capability address",
        doc="The CSP sub-array PST Inherent Capability FQDN.",
    )

    procModeVlbiAddr = attribute(
        dtype='DevString',
        label="VLBI Inhernt Capabilityaddress",
        doc="The CSP sub-array VLBI Inherent Capability FQDN.",
    )

    cbfSubarrayState = attribute(
        dtype='DevState',
    )

    pssSubarrayState = attribute(
        dtype='DevState',
    )

    cbfSubarrayHealthState = attribute(
        dtype='DevEnum',
        label="CBF Subarray Health State",
        doc="CBF Subarray Health State",
        enum_labels=["OK", "DEGRADED", "FAILED", "UNKNOWN", ],
    )

    pssSubarrayHealthState = attribute(
        dtype='DevEnum',
        label="PSS Subarray Health State",
        doc="PSS Subarray Health State",
        enum_labels=["OK", "DEGRADED", "FAILED", "UNKNOWN", ],
    )

    cbfSubarrayAdminMode = attribute(
        dtype='DevEnum',
        label="CBF Subarray Admin Mode",
        doc="CBF Subarray Admin Mode",
        enum_labels=["ON-LINE", "OFF-LINE", "MAINTENANCE", "NOT-FITTED", "RESERVED",],
    )

    pssSubarrayAdminMode = attribute(
        dtype='DevEnum',
        label="PSS Subarray Admin Mode",
        doc="PSS Subarray Admin Mode",
        enum_labels=["ON-LINE", "OFF-LINE", "MAINTENANCE", "NOT-FITTED", "RESERVED",],
    )
    cbfSubarrayObsState = attribute(
        dtype='DevEnum',
        label="CBF Subarray Observing State",
        doc="The CBF subarray observing state.",
        enum_labels=["IDLE", "CONFIGURING", "READY", "SCANNING", "PAUSED", "ABORTED", "FAULT",],
    )

    pssSubarrayObsState = attribute(
        dtype='DevEnum',
        label="PSS Subarray Observing State",
        doc="The PSS subarray observing state.",
        enum_labels=["IDLE", "CONFIGURING", "READY", "SCANNING", "PAUSED", "ABORTED", "FAULT",],
    )

    pssSubarrayAddr = attribute(
        dtype='DevString',
        label="PSS sub-array address",
        doc="The PSS sub-element sub-array FQDN.",
    )

    cbfSubarrayAddr = attribute(
        dtype='DevString',
        label="CBF sub-array address",
        doc="The CBF sub-element sub-array FQDN.",
    )

    validScanConfiguration = attribute(
        dtype='DevString',
        label="Valid Scan Configuration",
        doc="Store the last valid scan configuration.",
    )

    addSearchBeamDurationExpected = attribute(
        dtype='DevUShort',
        label="AddSearchBeams command duration expected",
        doc="The duration expected (in sec) for the AddSearchBeams command.",
    )

    remSearchBeamsDurationExpected = attribute(
        dtype='DevUShort',
        label="RemoveSearchBeams command duration expected",
        doc="The duration expected (in sec) for the RemoveSearchBeams command.",
    )

    addSearchBeamDurationMeasured = attribute(
        dtype='DevUShort',
        label="AddSearchBeams command duration measured",
        doc="The duration measured (in sec) for the AddSearchBeams command.",
    )

    remSearchBeamsDurationMeasured = attribute(
        dtype='DevUShort',
        label="RemoveSearchBeams command duration measured",
        doc="The duration measured (in sec) for the RemoveSearchBeams command.",
    )


    addTimingBeamDurationExpected = attribute(
        dtype='DevUShort',
        label="AddTimingBeams command duration expected",
        doc="The duration expected (in sec) for the AddTimingBeams command.",
    )

    remTimingBeamsDurationExpected = attribute(
        dtype='DevUShort',
        label="RemoveTimingBeams command duration expected",
        doc="The duration expected (in sec) for the RemoveTimingBeams command.",
    )

    addTimingBeamDurationMeasured = attribute(
        dtype='DevUShort',
        label="AddTimingBeams command duration measured",
        doc="The duration measured (in sec) for the AddTimingBeams command.",
    )

    remTimingBeamsDurationMeasured = attribute(
        dtype='DevUShort',
        label="EndSB command duration measured",
        doc="The duration measured (in sec) for the RemoveTimingBeams command.",
    )
 
    endSBDurationExpected = attribute(
        dtype='DevUShort',
        label="EndSB command duration expected",
        doc="The duration expected (in sec) for the EndSB command.",
    )

    endSBDurationMeasured = attribute(
        dtype='DevUShort',
        label="EndSB command duration measured",
        doc="The duration measured (in sec) for the EndSB command.",
    )

    endSBCmdProgress = attribute(
        dtype='DevUShort',
        label="EndSB command progress percentage",
        polling_period=1500,
        abs_change=5,
        doc="The progress percentage for the EndSB command.",
    )

    endScanDurationExpected = attribute(
        dtype='DevUShort',
        label="EndScan command duration expected",
        doc="The duration expected (in sec) for the EndScan command.",
    )

    endScanDurationMeasured = attribute(
        dtype='DevUShort',
        label="EndScan command duration measured",
        doc="The duration measured (in sec) for the EndScan command.",
    )

    endScanCmdProgress = attribute(
        dtype='DevUShort',
        label="EndScan command progress percentage",
        polling_period=1500,
        abs_change=5,
        doc="The progress percentage for the EndScan command.",
    )

    
    addResourcesCmdProgress = attribute(
        dtype='DevUShort',
        label="Add resources command progress percentage",
        polling_period=1500,
        abs_change=5,
        doc="The progress percentage for the Add resources command.",
    )
    
    removeResourcesCmdProgress = attribute(
        dtype='DevUShort',
        label="Remove resources command progress percentage",
        polling_period=1500,
        abs_change=5,
        doc="The progress percentage for the Add/SearchBeams command.",
    )
    
    scanCmdProgress = attribute(
        dtype='DevUShort',
        label="Scan command progress percentage",
        polling_period=1500,
        abs_change=5,
        doc="The progress percentage for the Scan command.",
    )

    reservedSearchBeamNum = attribute(
        dtype='DevUShort',
        label="Number of reserved SearchBeam IDs",
        doc="Number of SearchBeam IDs reserved for the CSP sub-array",
    )

    numOfDevCompletedTask = attribute(
        dtype='DevUShort',
        label="Number of devices that completed the task",
        doc="Number of devices that completed the task",
    )

    cmdTimeoutExpired = attribute(
        dtype='DevBoolean',
        label="CspSubarray command execution timeout flag",
        polling_period=1000,
        doc="The timeout flag for a CspSubarray command.",
    )

    cmdAlarmRaised = attribute(
        dtype='DevBoolean',
        label="CspSubarray alarm flag",
        polling_period=1000,
        doc="The alarm flag for a CspSubarray command.",
    )


    pstOutputLink = attribute(
        dtype='DevString',
        label="PST output link",
        doc="The output link for PST products.",
    )

    assignedSearchBeamIDs = attribute(
        dtype=('DevUShort',),
        max_dim_x=1500,
        label="List of assigned Search Beams",
        doc="List of assigned Search Beams",
    )

    reservedSearchBeamIDs = attribute(
        dtype=('DevUShort',),
        max_dim_x=1500,
        label="List of reserved SearchBeam IDs",
        doc="List of SearchBeam IDs reserved for the CSP sub-array",
    )

    assignedTimingBeamIDs = attribute(
        dtype=('DevUShort',),
        max_dim_x=16,
        label="List of assigned TimingBeam IDs",
        doc="List of TimingBeam IDs assigned to the  CSP sub-array",
    )

    assignedVlbiBeamIDs = attribute(
        dtype=('DevUShort',),
        max_dim_x=20,
        label="List of assigned VlbiBeam IDs",
        doc="List of VlbiBeam IDs assigned to the  CSP sub-array",
    )

    assignedSearchBeamsState = attribute(
        dtype=('DevState',),
        max_dim_x=1500,
        label="Assigned SearchBeams State",
        doc="State of the assigned SearchBeams",
    )

    assignedTimingBeamsState = attribute(
        dtype=('DevState',),
        max_dim_x=16,
        label="Assigned TimingBeams State",
        doc="State of the assigned TimingBeams",
    )

    assignedVlbiBeamsState = attribute(
        dtype=('DevState',),
        max_dim_x=20,
        label="Assigned VlbiBeams State",
        doc="State of the assigned VlbiBeams",
    )

    assignedSearchBeamsHealthState = attribute(
        dtype=('DevState',),
        max_dim_x=1500,
        label="Assigned SearchBeams HealthState",
        doc="HealthState of the assigned SearchBeams",
    )

    assignedTimingBeamsHealthState = attribute(
        dtype=('DevState',),
        max_dim_x=16,
        label="Assigned TimingBeams HealthState",
        doc="HealthState of the assigned TimingBeams",
    )

    assignedVlbiBeamsHealthState = attribute(
        dtype=('DevState',),
        max_dim_x=20,
        label="Assigned VlbiBeams HealthState",
        doc="HealthState of the assigned VlbiBeams",
    )

    assignedSearchBeamsObsState = attribute(
        dtype=('DevState',),
        max_dim_x=1500,
        label="Assigned SearchBeams ObsState",
        doc="ObsState of the assigned SearchBeams",
    )

    assignedTimingBeamsObsState = attribute(
        dtype=('DevState',),
        max_dim_x=16,
        label="Assigned TimingBeams ObsState",
        doc="ObsState of the assigned TimingBeams",
    )

    assignedVlbiBeamsObsState = attribute(
        dtype=('DevState',),
        max_dim_x=20,
        label="Assigned VlbiBeams ObsState",
        doc="ObsState of the assigned VlbiBeams",
    )

    assignedSearchBeamsAdminMode = attribute(
        dtype=('DevState',),
        max_dim_x=1500,
        label="Assigned SearchBeams AdminMode",
        doc="AdminMode of the assigned SearchBeams",
    )

    assignedTimingBeamsAdminMode = attribute(
        dtype=('DevState',),
        max_dim_x=16,
        label="Assigned TimingBeams AdminMode",
        doc="AdminMode of the assigned TimingBeams",
    )

    assignedVlbiBeamsAdminMode = attribute(
        dtype=('DevState',),
        max_dim_x=20,
        label="Assigned VlbiBeams AdminMode",
        doc="AdminMode of the assigned VlbiBeams",
    )

    pstBeams = attribute(
        dtype=('DevString',),
        max_dim_x=16,
        label="PSTBeams addresses",
        doc="PST sub-element PSTBeam TANGO device FQDNs.",
    )

    listOfDevCompletedTask = attribute(
        dtype=('DevString',),
        max_dim_x=100,
        label="List of devices that completed the task",
        doc="List of devices that completed the task",
    )

    cbfOutputLink = attribute(name="cbfOutputLink",
        label="cbfOutputLink",
        forwarded=True
    )
    #pssOutputLink = attribute(name="pssOutputLink",
    #    label="cbfOutputLink",
    #    forwarded=True
    #)
 
    #vlbiOutputLink = attribute(name="vlbiOutputLink",
    #    label="cbfOutputLink",
    #    forwarded=True
    #)
    # ---------------
    # General methods
    # ---------------

    def init_device(self):
        """Initialises the attributes and properties of the CspSubarray."""
        SKASubarray.init_device(self)
        # PROTECTED REGION ID(CspSubarray.init_device) ENABLED START #
        self._build_state = '{}, {}, {}'.format(release.name, release.version, release.description)
        self._version_id = release.version
        # connect to CSP.LMC TANGO DB
        self.set_state(tango.DevState.INIT)
        self._health_state = HealthState.UNKNOWN
        self._admin_mode = AdminMode.ONLINE
        self._obs_mode = ObsMode.IDLE
        self._obs_state = ObsState.IDLE
        # connect to TANGO DB
        
        # use defaultdict to initialize the sub-element State,healthState
        # and adminMode. The dictionary uses as keys the sub-element
        # fqdn, for example
        # self._sc_subarray_state[self.CspCbf]
        # return the State value of the Mid Cbf sub-element.
        self._sc_subarray_state         = defaultdict(lambda: tango.DevState.DISABLE)
        self._sc_subarray_health_state  = defaultdict(lambda: HealthState.UNKNOWN)
        self._sc_subarray_admin_mode    = defaultdict(lambda: AdminMode.NOTFITTED)
        self._sc_subarray_obs_state     = defaultdict(lambda: ObsState.IDLE)
        self._sc_subarray_obs_mode      = defaultdict(lambda: ObsMode.IDLE)
        self._csp_tango_db = tango.Database()
        # read the CSP memorized attributes from the TANGO DB. 
        # Note: a memorized attribute has defined the attribute
        # property '__value'
        attribute_properties = self._csp_tango_db.get_device_attribute_property(self.get_name(),
                                                                          {'adminMode': ['__value']})
        # build a dictionary with the (attr_name, value) of the memorized attributes
        # use memorized atrtibute if present, otherwise the default one
        memorized_attr_dict = {attr_name : int(value[0]) for attr_name, db_key in attribute_properties.items() 
                                                         for key, value in db_key.items() 
                                                         if key == '__value'}
        try:
            self._admin_mode = memorized_attr_dict['adminMode']
            if self._admin_mode not in [AdminMode.ONLINE, AdminMode.MAINTENANCE]:
                self._health_state = HealthState.UNKNOWN
                self.set_state(tango.DevState.DISABLE)
                
        except KeyError as key_err:
            self.logger.info("Key {} not found".format(key_err))

        # list of sub-array sub-component FQDNs
        self._sc_subarray_fqdn = []
        
        # list of sub-component FQDNs assigned to the sub-array
        self._sc_subarray_assigned_fqdn = []
        
        # _sc_subarray_proxies: the sub-element sub-array proxies
        # implementes as dictionary:
        # keys: sub-element sub-arrayFQDN
        # values: device proxy
        self._sc_subarray_proxies = {}
        
        # Nested default dictionary with list of event ids/CSP sub-array sub-component. Need to
        # store the event ids for each CSP sub-array component and attribute to un-subscribe
        # them at disconnection.
        # keys: sub-component FQDN
        # values: dictionary (keys: attribute name, values: event id)
        self._sc_subarray_event_id = defaultdict(lambda: defaultdict(lambda: 0))
        
        # _sc_subarray_cmd_exec_state: implement the execution state of a long-running
        # command for each sub-array sub-component.
        # implemented as a nested default dictionary:
        # keys: sub-element FQDN
        # values: defaut dictionary (keys: command name, values: command state)
        self._sc_subarray_cmd_exec_state = defaultdict(lambda: defaultdict(lambda: CmdExecState.IDLE))
        
        # _sc_subarray_cmd_starting_time: for each sub-element report the long-running command 
        # starting time
        # Implemented as dictionary:
        # keys: the sub-element sub-array FQDN
        # values: starting time
        self._sc_subarray_cmd_starting_time = defaultdict(lambda: 0.0)
        
        # _command_thread: thread for the command execution
        # keys: the command name('on, 'off'...)
        # values: thread instance
        self._command_thread = {}
        
        # of a long-running command
        # implemented as a default nested dictionary:
        # keys: sub-element sub-array FQDN
        # values: default dictionary (keys: command name, values: the execution percentage)
        self._sc_subarray_cmd_progress = defaultdict(lambda: defaultdict(lambda: 0))
        
        # _sc_subarray_cmd_duration_expected: for each sub-element, store the duration (in sec.) 
        # configured for a long-running command 
        # Implemented as a nested default dictionary
        # keys: FQDN
        # values: default dictionary (keys: command name, values: the duration (in sec))
        self._sc_subarray_cmd_duration_expected = defaultdict(lambda: defaultdict(lambda: 20))
        
        # _sc_subarray_scan_configuration: report the scan configuration
        # for each sub-array sub-component (CBF, PSS subarray, PSTBeams) 
        # Implemented as default dictionary
        # keys: FQDN
        # values: the scan confiration as JSON formatted string
        
        self._sc_subarray_scan_configuration = defaultdict(lambda:'')
        
        # _cmd_execution_state: implement the execution state of a long-running
        # command for the whole CSP sub-array  Setting this attribute prevent the execution
        # of the same command while it is already running.
        # implemented as a default dictionary:
        # keys: command name
        # values:command state
        self._cmd_execution_state = defaultdict(lambda: CmdExecState.IDLE)
        
        # _cmd_progress: report the execution progress of a long-running command
        # implemented as a dictionary:
        # keys: command name ('addbeams', 'removebeams', 'configure')
        # values: the percentage
        self._cmd_progress = defaultdict(lambda: 0)
        
        # _cmd_duration_expected: store the duration (in sec.) configured for
        # a long-running command 
        # Implemented asdefault dictionary
        # keys: command name ('on', 'off',..)
        # values: the duration (in sec)
        self._cmd_duration_expected = defaultdict(lambda: 30)
        
        # _cmd_duration_measured: report the measured duration (in sec.) for
        # a long-running command 
        # Implemented as default dictionary
        # keys: command name ('on', 'off',..)
        # values: the duration (in sec)
        self._cmd_duration_measured = defaultdict(lambda: 0)
        
        # _timeout_expired: report the timeout flag 
        self._timeout_expired = False
        
        # _alarm_raised: report the alarm flag 
        self._alarm_raised = False
        
         # _alarm_message: report the alarm message
        # Implemented as a dictionary
        # keys: command name ('addserachbeams', 'configurescan'..)
        # values: the message
        self._alarm_message = defaultdict(lambda: '')
        
        # _list_dev_completed_task: for each long-running command report the list of subordinate
        # components that completed the task
        # Implemented as a dictionary
        # keys: the command name ('on', 'off',...)
        # values: the list of components
        self._list_dev_completed_task = defaultdict(lambda: [])
        # the last executed command 
        self._last_executed_command = ""
         
        # _num_dev_completed_task: for each long-running command report the number
        #  of subordinate components that completed the task
        # Implemented as a dictionary
        # keys: the command name ('addbeams, 'configurescan',...)
        # values: the number of components
        self._num_dev_completed_task = defaultdict(lambda:0)
        
        # _valid_scan_configuration: the last programmed scan configuration
        self._valid_scan_configuration = ''
        
        # unassigned CSP SearchBeam, TimingBeam, VlbiBeam Capability IDs
        self._assigned_search_beams = []
        #self._unassigned_search_beam_num =  0
        self._reserved_search_beams =  []
        #self._reserved_search_beam_num =  0
        self._assigned_timing_beams= []
        self._assigned_vlbi_beams = []      
                                                           
        # Try connection with the CBF sub-array
        self._connect_to_subarray_subcomponent(self.CbfSubarray)
        # Try connection with the PSS sub-array
        self._connect_to_subarray_subcomponent(self.PssSubarray)
                
        # to use the push model in command_inout_asynch (the one with the callback parameter),
        # change the global TANGO model to PUSH_CALLBACK.
        apiutil = tango.ApiUtil.instance()
        apiutil.set_asynch_cb_sub_model(tango.cb_sub_model.PUSH_CALLBACK)
        # PROTECTED REGION END #    //  CspSubarray.init_device

    def always_executed_hook(self):
        """Method always executed before any TANGO command is executed."""
        # PROTECTED REGION ID(CspSubarray.always_executed_hook) ENABLED START #
        # PROTECTED REGION END #    //  CspSubarray.always_executed_hook

    def delete_device(self):
        """Hook to delete resources allocated in init_device.

        This method allows for any memory or other resources allocated in the
        init_device method to be released.  This method is called by the device
        destructor and by the device Init command.
        """
        # PROTECTED REGION ID(CspSubarray.delete_device) ENABLED START #
        #release the allocated event resources
        for fqdn in self._sc_subarray_fqdn:
            event_to_remove = []
            try:
                for event_id in self._sc_subarray_event_id[fqdn]:
                    try:
                        self._sc_subarrays_proxies[fqdn].unsubscribe_event(event_id)
                        # self._se_subarray_event_id[fqdn].remove(event_id)
                        # in Pyton3 can't remove the element from the list while looping on it.
                        # Store the unsubscribed events in a temporary list and remove them later.
                        event_to_remove.append(event_id)
                    except tango.DevFailed as df:
                        msg = ("Unsubscribe event failure.Reason: {}. "
                               "Desc: {}".format(df.args[0].reason, df.args[0].desc))
                        self.logger.error(msg)
                    except KeyError as key_err:
                        # NOTE: in PyTango unsubscription of a not-existing event id raises a
                        # KeyError exception not a DevFailed!!
                        msg = "Unsubscribe event failure. Reason: {}".format(str(key_err))
                        self.dev_logging(msg, tango.LogLevel.LOG_ERROR)
                # remove the events from the list
                for k in event_to_remove:
                    self._se_subarray_event_id[fqdn].remove(k)
                # check if there are still some registered events. What to do in this case??
                if self._se_subarray_event_id[fqdn]:
                    msg = "Still subscribed events: {}".format(self._se_subarray_event_id)
                    self.dev_logging(msg, tango.LogLevel.LOG_WARN)
                else:
                    # delete the dictionary entry
                    self._se_subarray_event_id.pop(fqdn)
            except KeyError as key_err:
                msg = " Can't retrieve the information of key {}".format(key_err)
                self.dev_logging(msg, tango.LogLevel.LOG_ERROR)
        # clear the subarrays list and dictionary
        self._se_subarrays_fqdn.clear()
        self._se_subarrays_proxies.clear()

        # PROTECTED REGION END #    //  CspSubarray.delete_device
    # ------------------
    # Attributes methods
    # ------------------

    def read_scanID(self):
        # PROTECTED REGION ID(CspSubarray.scanID_read) ENABLED START #
        """Return the scanID attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.scanID_read

    def write_scanID(self, value):
        # PROTECTED REGION ID(CspSubarray.scanID_write) ENABLED START #
        """Set the scanID attribute."""
        pass
        # PROTECTED REGION END #    //  CspSubarray.scanID_write

    def read_procModeCorrelationAddr(self):
        # PROTECTED REGION ID(CspSubarray.corrInherentCapAddr_read) ENABLED START #
        """Return the procModeCorrelationAddr attribute."""
        return self.SubarrayProcModeCorrelation
        # PROTECTED REGION END #    //  CspSubarray.corrInherentCapAddr_read

    def read_procModePssAddr(self):
        # PROTECTED REGION ID(CspSubarray.pssInherentCapAddr_read) ENABLED START #
        """Return the procModePssAddr attribute."""
        return self.SubarrayProcModePss
        # PROTECTED REGION END #    //  CspSubarray.pssInherentCapAddr_read

    def read_procModePstAddr(self):
        # PROTECTED REGION ID(CspSubarray.pstInherentCap_read) ENABLED START #
        """Return the procModePstAddr( attribute."""
        return self.SubarrayProcModePst
        # PROTECTED REGION END #    //  CspSubarray.pstInherentCap_read

    def read_procModeVlbiAddr(self):
        # PROTECTED REGION ID(CspSubarray.vlbiInherentCap_read) ENABLED START #
        """Return the procModeVlbiAddr attribute."""
        return self.SubarrayProcModeVlbi
        # PROTECTED REGION END #    //  CspSubarray.vlbiInherentCap_read

    def read_cbfSubarrayState(self):
        # PROTECTED REGION ID(CspSubarray.cbfSubarrayState_read) ENABLED START #
        """Return the cbfSubarrayState attribute."""
        return self._sc_subarray_state[self.CbfSubarray]
        # PROTECTED REGION END #    //  CspSubarray.cbfSubarrayState_read

    def read_pssSubarrayState(self):
        # PROTECTED REGION ID(CspSubarray.pssSubarrayState_read) ENABLED START #
        """Return the pssSubarrayState attribute."""
        return self._sc_subarray_state[selPssSubarray]
        # PROTECTED REGION END #    //  CspSubarray.pssSubarrayState_read

    def read_cbfSubarrayHealthState(self):
        # PROTECTED REGION ID(CspSubarray.cbfSubarrayHealthState_read) ENABLED START #
        """Return the cbfSubarrayHealthState attribute."""
        return self._sc_subarray_health_state[self.CbfSubarray]
        # PROTECTED REGION END #    //  CspSubarray.cbfSubarrayHealthState_read

    def read_pssSubarrayHealthState(self):
        # PROTECTED REGION ID(CspSubarray.pssSubarrayHealthState_read) ENABLED START #
        """Return the pssSubarrayHealthState attribute."""
        return self._sc_subarray_health_state[self.PssSubarray]
        # PROTECTED REGION END #    //  CspSubarray.pssSubarrayHealthState_read

    def read_cbfSubarrayAdminMode(self):
        # PROTECTED REGION ID(CspSubarray.cbfSubarrayHealthState_read) ENABLED START #
        """Return the cbfSubarrayHealthState attribute."""
        return self._sc_subarray_admin_mode[self.CbfSubarray]
        # PROTECTED REGION END #    //  CspSubarray.cbfSubarrayHealthState_read

    def read_pssSubarrayAdminMode(self):
        # PROTECTED REGION ID(CspSubarray.pssSubarrayHealthState_read) ENABLED START #
        """Return the pssSubarrayHealthState attribute."""
        return self._sc_subarray_admin_mode[self.PssSubarray]
        # PROTECTED REGION END #    //  CspSubarray.pssSubarrayHealthState_read
        
    def read_cbfSubarrayObsState(self):
        # PROTECTED REGION ID(CspSubarray.cbfSubarrayObsState_read) ENABLED START #
        """Return the cbfSubarrayObsState attribute."""
        return self._sc_subarray_obs_state[self.CbfSubarray]
        # PROTECTED REGION END #    //  CspSubarray.cbfSubarrayObsState_read

    def read_pssSubarrayObsState(self):
        # PROTECTED REGION ID(CspSubarray.pssSubarrayObsState_read) ENABLED START #
        """Return the pssSubarrayObsState attribute."""
        return self._sc_subarray_obs_state[self.PssSubarray]
        # PROTECTED REGION END #    //  CspSubarray.pssSubarrayObsState_read

    def read_pssSubarrayAddr(self):
        # PROTECTED REGION ID(CspSubarray.pssSubarrayAddr_read) ENABLED START #
        """Return the pssSubarrayAddr attribute."""
        return self.PssSubarray
        # PROTECTED REGION END #    //  CspSubarray.pssSubarrayAddr_read

    def read_cbfSubarrayAddr(self):
        # PROTECTED REGION ID(CspSubarray.cbfSubarrayAddr_read) ENABLED START #
        """Return the cbfSubarrayAddr attribute."""
        return self.CbfSubarray
        # PROTECTED REGION END #    //  CspSubarray.cbfSubarrayAddr_read

    def read_validScanConfiguration(self):
        # PROTECTED REGION ID(CspSubarray.validScanConfiguration_read) ENABLED START #
        """Return the validScanConfiguration attribute."""
        return self._valid_scan_configuration
        # PROTECTED REGION END #    //  CspSubarray.validScanConfiguration_read

    def read_addSearchBeamDurationExpected(self):
        # PROTECTED REGION ID(CspSubarray.addSearchBeamDurationExpected_read) ENABLED START #
        """Return the addSearchBeamDurationExpected attribute."""
        return self._cmd_duration_expected['addsearchbeam']
        # PROTECTED REGION END #    //  CspSubarray.addSearchBeamDurationExpected_read

    def read_remSearchBeamsDurationExpected(self):
        # PROTECTED REGION ID(CspSubarray.remSearchBeamsDurationExpected_read) ENABLED START #
        """Return the remSearchBeamsDurationExpected attribute."""
        return self._cmd_duration_expected['rmsearchbeam']
        # PROTECTED REGION END #    //  CspSubarray.remSearchBeamsDurationExpected_read

    def read_addSearchBeamDurationMeasured(self):
        # PROTECTED REGION ID(CspSubarray.addSearchBeamDurationMeasured_read) ENABLED START #
        """Return the addSearchBeamDurationMeasured attribute."""
        return self._cmd_duration_measured['addsearchbeam']
        # PROTECTED REGION END #    //  CspSubarray.addSearchBeamDurationMeasured_read

    def read_remSearchBeamsDurationMeasured(self):
        # PROTECTED REGION ID(CspSubarray.remSearchBeamsDurationMeasured_read) ENABLED START #
        """Return the remSearchBeamsDurationMeasured attribute."""
        return self._cmd_duration_measured['rmsearchbeam']
        # PROTECTED REGION END #    //  CspSubarray.remSearchBeamsDurationMeasured_read

    def read_addTimingBeamDurationExpected(self):
        # PROTECTED REGION ID(CspSubarray.addTimingBeamDurationExpected_read) ENABLED START #
        """Return the addTimingBeamDurationExpected attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.addTimingBeamDurationExpected_read

    def read_remTimingBeamsDurationExpected(self):
        # PROTECTED REGION ID(CspSubarray.remTimingBeamsDurationExpected_read) ENABLED START #
        """Return the remTimingBeamsDurationExpected attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.remTimingBeamsDurationExpected_read

    def read_addTimingBeamDurationMeasured(self):
        # PROTECTED REGION ID(CspSubarray.addTimingBeamDurationMeasured_read) ENABLED START #
        """Return the addTimingBeamDurationMeasured attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.addTimingBeamDurationMeasured_read

    def read_remTimingBeamsDurationMeasured(self):
        # PROTECTED REGION ID(CspSubarray.remTimingBeamsDurationMeasured_read) ENABLED START #
        """Return the remTimingBeamsDurationMeasured attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.remTimingBeamsDurationMeasured_read
    
    def read_addResourcesCmdProgress(self):
        # PROTECTED REGION ID(CspSubarray.searchBeamsCmdProgress_read) ENABLED START #
        """Return the assResourcesCmdProgress attribute."""
        return self._cmd_progress['addbeam']
        # PROTECTED REGION END #    //  CspSubarray.searchBeamsCmdProgress_read

    def read_removeResourcesCmdProgress(self):
        # PROTECTED REGION ID(CspSubarray.timingBeamsCmdProgress_read) ENABLED START #
        """Return the removeResourcesCmdProgress attribute."""
        return self._cmd_progress['removebeams']
        # PROTECTED REGION END #    //  CspSubarray.timingBeamsCmdProgress_read
        
    def read_scanCmdProgress(self):
        # PROTECTED REGION ID(CspSubarray.endSBCmdProgress_read) ENABLED START #
        """Return the ScanCmdProgress attribute."""
        return self._cmd_progress['scan']
        # PROTECTED REGION END #    //  CspSubarray.endSBCmdProgress_read

    def read_endSBDurationExpected(self):
        # PROTECTED REGION ID(CspSubarray.endSBDurationExpected_read) ENABLED START #
        """Return the endSBDurationExpected attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.endSBDurationExpected_read

    def read_endSBDurationMeasured(self):
        # PROTECTED REGION ID(CspSubarray.endSBDurationMeasured_read) ENABLED START #
        """Return the endSBDurationMeasured attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.endSBDurationMeasured_read

    def read_endSBCmdProgress(self):
        # PROTECTED REGION ID(CspSubarray.endSBCmdProgress_read) ENABLED START #
        """Return the endSBCmdProgress attribute."""
        return self._cmd_progress['endsb']
        # PROTECTED REGION END #    //  CspSubarray.endSBCmdProgress_read

    def read_endScanDurationExpected(self):
        # PROTECTED REGION ID(CspSubarray.endScanDurationExpected_read) ENABLED START #
        """Return the endScanDurationExpected attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.endScanDurationExpected_read

    def read_endScanDurationMeasured(self):
        # PROTECTED REGION ID(CspSubarray.endScanDurationMeasured_read) ENABLED START #
        """Return the endScanDurationMeasured attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.endScanDurationMeasured_read

    def read_endScanCmdProgress(self):
        # PROTECTED REGION ID(CspSubarray.endScanCmdProgress_read) ENABLED START #
        """Return the endScanCmdProgress attribute."""
        return self._cmd_progress['endscan']
        # PROTECTED REGION END #    //  CspSubarray.endScanCmdProgress_read

    def read_reservedSearchBeamNum(self):
        # PROTECTED REGION ID(CspSubarray.reservedSearchBeamNum_read) ENABLED START #
        """Return the reservedSearchBeamNum attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.reservedSearchBeamNum_read

    def read_numOfDevCompletedTask(self):
        # PROTECTED REGION ID(CspSubarray.numOfDevCompletedTask_read) ENABLED START #
        """Return the numOfDevCompletedTask attribute."""
        return 0
        # PROTECTED REGION END #    //  CspSubarray.numOfDevCompletedTask_read

    def read_assignedSearchBeamIDs(self):
        # PROTECTED REGION ID(CspSubarray.assignedSearchBeamIDs_read) ENABLED START #
        """Return the assignedSearchBeamIDs attribute."""
        return _assigned_search_beams
        # PROTECTED REGION END #    //  CspSubarray.assignedSearchBeamIDs_read

    def read_reservedSearchBeamIDs(self):
        # PROTECTED REGION ID(CspSubarray.reservedSearchBeamIDs_read) ENABLED START #
        """Return the reservedSearchBeamIDs attribute."""
        return _reserved_search_beams
        # PROTECTED REGION END #    //  CspSubarray.reservedSearchBeamIDs_read

    def read_assignedTimingBeamIDs(self):
        # PROTECTED REGION ID(CspSubarray.assignedTimingBeamIDs_read) ENABLED START #
        """Return the assignedTimingBeamIDs attribute."""
        return _assigned_timing_beams
        # PROTECTED REGION END #    //  CspSubarray.assignedTimingBeamIDs_read

    def read_assignedVlbiBeamIDs(self):
        # PROTECTED REGION ID(CspSubarray.assignedVlbiBeam IDs_read) ENABLED START #
        """Return the assignedVlbiBeam IDs attribute."""
        return _assigned_vlbi_beams
        # PROTECTED REGION END #    //  CspSubarray.assignedVlbiBeam IDs_read

    def read_assignedSearchBeamsState(self):
        # PROTECTED REGION ID(CspSubarray.assignedSearchBeamsState_read) ENABLED START #
        """Return the assignedSearchBeamsState attribute."""
        self._search_beams_state = {key:value for key,value in self._subarray_resources_state.items()
                                           if key.find("search") > -1}
        return self._search_beams_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedSearchBeamsState_read

    def read_assignedTimingBeamsState(self):
        # PROTECTED REGION ID(CspSubarray.assignedTimingBeamsState_read) ENABLED START #
        """Return the assignedTimingBeamsState attribute."""
        self._timing_beams_state = {key:value for key,value in self._subarray_resources_state.items()
                                    if key.find("timing") > -1}
        return self._timing_beams_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedTimingBeamsState_read

    def read_assignedVlbiBeamsState(self):
        # PROTECTED REGION ID(CspSubarray.assignedVlbiBeamsState_read) ENABLED START #
        """Return the assignedVlbiBeamsState attribute."""
        self._vlbi_beams_state = {key:value for key,value in self._subarray_resources_state.items()
                                  if key.find("vlbi") > -1}
        return self._vlbi_beams_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedVlbiBeamsState_read

    def read_assignedSearchBeamsHealthState(self):
        # PROTECTED REGION ID(CspSubarray.assignedSearchBeamsHealthState_read) ENABLED START #
        """Return the assignedSearchBeamsHealthState attribute."""
        self._search_beams_health_state = {key:value for key,value in
                                           self._subarray_resources_health_state.item()
                                           if key.find("search") > -1}
        return self._search_beams_health_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedSearchBeamsHealthState_read

    def read_assignedTimingBeamsHealthState(self):
        # PROTECTED REGION ID(CspSubarray.assignedTimingBeamsHealthState_read) ENABLED START #
        """Return the assignedTimingBeamsHealthState attribute."""
        self._timing_beams_health_state = {key:value for key,value in
                                           self._subarray_resources_state.items()
                                           if key.find("timing") > -1}
        return self._timing_beams_health_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedTimingBeamsHealthState_read

    def read_assignedVlbiBeamsHealthState(self):
        # PROTECTED REGION ID(CspSubarray.assignedVlbiBeamsHealthState_read) ENABLED START #
        """Return the assignedVlbiBeamsHealthState attribute."""
        self._vlbi_beams_health_state = {key:value for key,value
                                         in self._subarray_resources_state.items()
                                         if key.find("vlbi") > -1}
        return self._vlbi_beams_health_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedVlbiBeamsHealthState_read

    def read_assignedSearchBeamsObsState(self):
        # PROTECTED REGION ID(CspSubarray.assignedSearchBeamsObsState_read) ENABLED START #
        """Return the assignedSearchBeamsObsState attribute."""
        self._search_beams_obs_state = {key:value for key,value in
                                  self._subarray_resources_health_state.items()
                                  if key.find("search") > -1}
        return self._search_beams_obs_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedSearchBeamsObsState_read

    def read_assignedTimingBeamsObsState(self):
        # PROTECTED REGION ID(CspSubarray.assignedTimingBeamsObsState_read) ENABLED START #
        """Return the assignedTimingBeamsObsState attribute."""
        self._timing_beams_obs_state = {key:value for key,value in
                                        self._subarray_resources_state.items()
                                        if key.find("timing") > -1}
        return self._timing_beams_obs_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedTimingBeamsObsState_read

    def read_assignedVlbiBeamsObsState(self):
        # PROTECTED REGION ID(CspSubarray.assignedVlbiBeamsObsState_read) ENABLED START #
        """Return the assignedVlbiBeamsObsState attribute."""
        self._vlbi_beams_obs_state = {key:value for key,value in
                                      self._subarray_resources_state.items()
                                      if key.find("vlbi") > -1}
        return self._vlbi_beams_obs_state.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedVlbiBeamsObsState_read

    def read_assignedSearchBeamsAdminMode(self):
        # PROTECTED REGION ID(CspSubarray.assignedSearchBeamsAdminMode_read) ENABLED START #
        """Return the assignedSearchBeamsAdminMode attribute."""
        self._search_beams_admin_mode = {key:value for key,value in
                                         self._subarray_resources_health_state.items()
                                         if key.find("search") > -1}
        return self._search_beams_admin_mode.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedSearchBeamsAdminMode_read

    def read_assignedTimingBeamsAdminMode(self):
        # PROTECTED REGION ID(CspSubarray.assignedTimingBeamsAdminMode_read) ENABLED START #
        """Return the assignedTimingBeamsAdminMode attribute."""
        self._timing_beams_admin_mode = {key:value for key,value in
                                         self._subarray_resources_state.items()
                                         if key.find("timing") > -1}
        return self._timing_beams_admin_mode.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedTimingBeamsAdminMode_read

    def read_assignedVlbiBeamsAdminMode(self):
        # PROTECTED REGION ID(CspSubarray.assignedVlbiBeamsAdminMode_read) ENABLED START #
        """Return the assignedVlbiBeamsAdminMode attribute."""
        self._vlbi_beams_admin_mode = {key:value for key,value in
                                       self._subarray_resources_state.items()
                                       if key.find("vlbi") > -1}
        return self._vlbi_beams_admin_mode.values()
        # PROTECTED REGION END #    //  CspSubarray.assignedVlbiBeamsAdminMode_read

    def read_pstBeams(self):
        # PROTECTED REGION ID(CspSubarray.pstBeams_read) ENABLED START #
        """Return the pstBeams attribute."""
        return self.PstBeams
        # PROTECTED REGION END #    //  CspSubarray.pstBeams_read
        
    def read_cmdTimeoutExpired(self):
        # PROTECTED REGION ID(CspSubarray.cmdTimeoutExpired_read) ENABLED START #
        """Return the cmdTimeoutExpired attribute."""
        return False
        # PROTECTED REGION END #    //  CspSubarray.cmdTimeoutExpired_read

    def read_cmdAlarmRaised(self):
        # PROTECTED REGION ID(CspSubarray.cmdAlarmRaised_read) ENABLED START #
        """Return the cmdAlarmRaised attribute."""
        return False
        # PROTECTED REGION END #    //  CspSubarray.cmdAlarmRaised_read

    def read_pstOutputLink(self):
        # PROTECTED REGION ID(CspSubarray.pstOutputLink_read) ENABLED START #
        """Return the pstOutputLink attribute."""
        return ''
        # PROTECTED REGION END #    //  CspSubarray.pstOutputLink_read
    def read_listOfDevCompletedTask(self):
        # PROTECTED REGION ID(CspSubarray.listOfDevCompletedTask_read) ENABLED START #
        """Return the listOfDevCompletedTask attribute."""
        return ('',)
        # PROTECTED REGION END #    //  CspSubarray.listOfDevCompletedTask_read


    # --------
    # Commands
    # --------
          
    @AdminModeCheck('EndScan')
    @ObsStateCheck('endscan')
    def is_EndScan_allowed(self):
        return self._is_subarray_configuring_allowed()   
                  
    @command(
    )
    @DebugIt()
    def EndScan(self):
        # PROTECTED REGION ID(CspSubarray.EndScan) ENABLED START #
        """
        *Class method*
        End the execution of a running scan. After successful execution, the CspSubarray \
        *ObsState* is  IDLE.

        raises:
            tango.DevFailed: if the subarray *obsState* is not SCANNING or if an exception
            is caught during the command execution.
        Note:
            Still to implement the check on AdminMode values: the command can be processed \
            only when the CspSubarray is *ONLINE* or *MAINTENANCE*
        """
        for device in self._sc_subarray_fqdn:
            try:
                proxy = self._sc_subarray_proxies[device]
            except KeyError as key_err:
                self.logger.warn("No key {} found".format(key_err)) 
            proxy.command_inout_asynch("EndScan", self._cmd_ended_cb)
        # PROTECTED REGION END #    //  CspSubarray.EndScan
    
    @AdminModeCheck('Scan')
    @ObsStateCheck('scan')
    def is_Scan_allowed(self):
        return self._is_subarray_configuring_allowed()   
                            
    @command(
        dtype_in='DevVarStringArray',
    )
    @DebugIt()
    def Scan(self, argin):
        # PROTECTED REGION ID(CspSubarray.Scan) ENABLED START #
        """
        Starts the scan.

        :param argin: 'DevVarStringArray'

        :return:None
        """
        # invoke the constructor for the command thread
        self._cmd_progress['scan'] = 0
        for device in self._sc_subarray_fqdn:
            try:
                proxy = self._sc_subarray_proxies[device]
                if not self._sc_subarray_event_id[device]['scancmdprogress']:
                    evt_id = proxy.subscribe_event("scanCmdProgress",
                                                   tango.EventType.CHANGE_EVENT,
                                                   self._sc_attributes_change_event_cb,
                                                   stateless=False)
                    self._sc_subarray_event_id[device]['scancmdprogress'] = evt_id
            except KeyError as key_err:
                self.logger.warn("No key {} found".format(key_err)) 
            except tango.DevFailed as tango_err:
                self.logger.info(tango_err.args[0].desc)
            proxy.command_inout_asynch("Scan", self._cmd_ended_cb)
            
        # PROTECTED REGION END #    //  CspSubarray.Scan    
    def __validate_scan_configuration(self, json_config):
        """
        here is done a preliminary validation of the configuraito:
        - look for the sub-element entries
        - add eventual information to the sub-element blocks (for example for PSS add the
        addresses of the PSS pipelines or the IDs of the SearchBeams)
        - set the observing mode for each sub-element sub-array/PSTBeam
    
        if "cbf" in json_config:
            self._sc_subarray_obs_mode[self.CbfSubarray] = ObsMode.CORRELATION
            scan_configutation[self.CbfSubarray] = json_config["cbf"]
        if "pss" in json_config:
            scan_configutation[self.PssSubarray] = json_config["pss"]
        if "pst" in json_config:
            scan_configuration[self.PssSubarray] = json_config["pst"]
        return scan_configuration_dict
    
        Customize the function for each instance
    """
        if "cbf" in json_config:
            if self._sc_subarray_state[self.CbfSubarray] != tango.DevState.ON:
                pass
                # throw exception
            self._sc_subarray_obs_mode[self.CbfSubarray] = ObsMode.CORRELATION
            self._sc_scan_configuration[self.CbfSubarray] = json_config["cbf"]
            self._sc_subarray_assigned_fqdn.append(self.CbfSubarray)
        if "pss" in json_config:
            if self._sc_subarray_state[self.PssSubarray] != tango.DevState.ON:
                pass
                # throw exception
            #self._sc_subarray_obs_mode[self.PssSubarray] = ??
            self._sc_subarray_assigned_fqdn.append(self.PssSubarray)
            self._sc_scan_configuration[self.PssSubarray] = json_config["pss"]
            if "searchbeams" not in self._sc_scan_configuration[self.PssSubarray]:
                self._sc_scan_configuration[self.PssSubarray]["searchbeams"] = self._assigned_search_beams
        if "pst" in json_config:
            
            pass
        
    @AdminModeCheck('ConfigureScan')
    def is_ConfigureScan_allowed(self):
        return self._is_subarray_configuring_allowed()   
                     
    @command(
        dtype_in='DevString',
        doc_in="A Json-encoded string with the scan configuration.",
    )
    @DebugIt()
    @ObsStateCheck('configscan')
    @SubarrayRejectCmd(['AddSearchBeams',
                       'RemoveSearchBeams',
                       'AddTimingBeams',
                       'RemoveTimingBeams'
                       'AddNumOfSearchBeams',
                       'RemoveNumOfSearchBeams'])
    def ConfigureScan(self, argin):
        """
        *TANGO Class method*

        Configure a scan for the subarray.\n
        The command can be execuced when the CspSubarray State is *ON* and the \
        ObsState is *IDLE* or *READY*.\n
        If the configuration for the scan is not correct (invalid parameters or invalid JSON)\
        the configuration is not applied and the ObsState of the CspSubarray remains IDLE.

        :param argin: a JSON-encoded string with the parameters to configure a scan.
        :return: None
        :raises:
            tango.DevFailed exception if the CspSubarray ObsState is not valid or if an exception\
            is caught during command execution.
        Note:
            Still to implement the check on AdminMode values: the command can be processed \
            only when the CspSubarray is *ONLINE* or *MAINTENANCE*
        """
        # PROTECTED REGION ID(CspSubarray.ConfigureScan) ENABLED START #

        # checks on Stae, adminMode and obsState values are performed inside the 
        # python decorators
       
        # the dictionary with the scan configuration
        argin_dict = {}
        try:
            # for test purpose we load the json configuration from an
            # external file.
            if "load" in argin:
                # skip the 'load' chars and remove spaces from the filename
                fn = (argin[4:]).strip()
		# !!!Note!! set the path to the data file
                filename = os.path.join(commons_pkg_path, fn)
                with open(filename) as json_file:
                    # load the file into a dictionary
                    argin_dict = json.load(json_file)
                    # dump the dictionary into the input string to forward
                    # to CbfSubarray
                    argin = json.dumps(argin_dict)
            else:
                argin_dict = json.loads(argin)
        except FileNotFoundError as file_err:
            self.logger.error(file_err)
            tango.Except.throw_exception("Command failed",
                                         str(file_err),
                                         "ConfigureScan execution",
                                         tango.ErrSeverity.ERR)
        except json.JSONDecodeError as e:  # argument not a valid JSON object
            # this is a fatal error
            msg = ("Scan configuration object is not a valid JSON object."
                   "Aborting configuration:{}".format(str(e)))
            self.logger.error(msg)
            tango.Except.throw_exception("Command failed",
                                         msg,
                                         "ConfigureScan execution",
                                         tango.ErrSeverity.ERR)
        self._validate_scan_configuration(argin)
        
        # Forward the ConfigureScan command to CbfSubarray.
        #self._timeout_expired = False
        #self._alarm_raised = False
        for device in self._sc_subarray_assigned_fqdn:
            #json_config = json.dumps(self._sc_subarray_scan_configuration[device])
            #print("json_config:", json_config)
            proxy = self._sc_subarray_proxies[device]
            attributes = ['configureCmdProgress', 'configureCmdTimeoutExpired']
            for attr in attributes:
                try:
                    if self._sc_subarray_event_id[attr.lower()] == 0:
                        event_id = proxy.subscribe_event(attr,
                                                         tango.EventType.CHANGE_EVENT,
                                                         self._attributes_change_evt_cb,
                                                         stateless=False)
                        self._sc_subarray_event_id[device][attr] = event_id
                except tango.DevFailed as df:
                    self.logger.info(df.args[0].desc)                  
        args_dict = {'cmd_name':"ConfigureScan",
                     'obs_state':ObsState.READY}
        # invoke the constructor for the command thread
        self._command_thread['configurescan'] = threading.Thread(target=self.__configure_scan,
                                                           name="Thread-ConfigureScan",
                                                           args=(self._sc_subarray_assigned_fqdn,),
                                                           kwargs=args_dict)
        self._command_thread['configurescan'].start()
        # sleep for a while to let the thread start
        time.sleep(0.2)
        # PROTECTED REGION END #    //  CspSubarray.ConfigureScan
    
    @AdminModeCheck('AddNumOfSearchBeams')
    @ObsStateCheck('addresources')
    def is_AddNumOfSearchBeams_allowed(self):
        return self._is_subarray_composition_allowed()
        
    @command(
        dtype_in='DevUShort',
        doc_in="The number of SearchBeams Capabilities to assign to the subarray.",
    )
    @DebugIt()
    @SubarrayRejectCmd(['RemoveSearchBeams',
                        'RemoveNumOfSearchBeams',
                        'ConfigureScan'])
    def AddNumOfSearchBeams(self, argin):
        # PROTECTED REGION ID(CspSubarray.AddNumOfSearchBeams) ENABLED START #
        self._cmd_execution_state['addsearchbeams'] = CmdExecState.RUNNING
        # connect to CspMaster and check if enough beams are available
        # forward the command to PSS
        # wait for PSS number of assigned pipelines == number of required beams
        # get the PSS beams IDs
        # if map exists, retrieve the SearchBeam IDs
        # get the corresponding SearchBeams FQDNs
        # open proxies to devices
        # subscribe SearchBeams attributes
        # PROTECTED REGION END #    //  CspSubarray.AddNumOfSearchBeams
        pass
    
    @AdminModeCheck('RemoveNumOfSearchBeams')
    @ObsStateCheck('removeresources')
    def is_RemoveNumOfSearchBeams_allowed(self):
        return self._is_subarray_composition_allowed()
        
    @command(
        dtype_in='DevUShort',
        doc_in="The number of SearchBeam Capabilities to remove from the \nCSP sub-array.\nAll the assigned SearcBbeams are removed from the CSP sub-array if the\ninput number is equal to the max number of SearchBeam \nCapabilities for the specified Csp sub-array instance (1500 for MID,\n500 for LOW)",
    )
    @DebugIt()
    @SubarrayRejectCmd(['AddSearchBeams', 'ConfigureScan'])
    def RemoveNumOfSearchBeams(self, argin):
        # PROTECTED REGION ID(CspSubarray.RemoveNumOfSearchBeams) ENABLED START #
        """
        
            Remove the specified number of SearchBeam Capabilities
            from the subarray.

        :param argin: 'DevUShort'
        
            The number of SearchBeam Capabilities to remove from the 
            CSP sub-array.
            All the assigned SearcBbeams are removed from the CSP sub-array if the
            input number is equal to the max number of SearchBeam 
            Capabilities for the specified Csp sub-array instance (1500 for MID,
            500 for LOW)

        :return:None
        """
        pass
        # PROTECTED REGION END #    //  CspSubarray.RemoveNumOfSearchBeams

    
    @AdminModeCheck('AddTimingBeams')
    @ObsStateCheck('addresources')
    def is_AddTimingBeams_allowed(self):
        return self._is_subarray_composition_allowed()
    
    @command(
        dtype_in='DevVarUShortArray',
        doc_in="The list of TimingBeam Capability IDs to assign to the CSP sub-array.",
    )
    @DebugIt()
    @SubarrayRejectCmd(['RemoveTimingBeams', 'ConfigureScan'])
    def AddTimingBeams(self, argin):
        # PROTECTED REGION ID(CspSubarray.AddTimingBeams) ENABLED START #
        """
        
            Add the specified TimingBeam Capability IDs to the CSP
            sub-array.

        :param argin: 'DevVarUShortArray'
        The list of TimingBeam Capability IDs to assign to the CSP sub-array.

        :return:None
        """
        # read the  CSP Master list of CSP TimingBeams addresses
        # get the CSP TimingBeams addresses from IDs
        # map timing beams ids to PST Beams ids
        # write the CSP TimingBeams membership
        # get the PSTBeams addresses

        pass
        # PROTECTED REGION END #    //  CspSubarray.AddTimingBeams

    @AdminModeCheck('AddSearchBeams')
    @ObsStateCheck('addresources')
    def is_AddSearchBeams_allowed(self):
        return self._is_subarray_composition_allowed()
    
    @command(
        dtype_in='DevVarUShortArray',
        doc_in="The list of SearchBeam Capability IDs to assign to the CSP sub-array.",
    )
    @DebugIt()
    @SubarrayRejectCmd(['RemoveSearchBeams', 'ConfigureScan'])
    def AddSearchBeams(self, argin):
        # PROTECTED REGION ID(CspSubarray.AddSearchBeams) ENABLED START #
        """
        
            Add the specified SeachBeam Capability IDs o the CSP 
            sub-array.

        :param argin: 'DevVarUShortArray'
        The list of SearchBeam Capability IDs to assign to the CSP sub-array.

        :return:None
        """
        # read the  CSP Master list of CSP SearchBeams addresses
        # get the CSP SearchBeams addresses from IDs
        # map search beams ids to PSS beams IDs
        # issue the AddSearchBeams command on PSS sub-array
        # wait until PSS list of beams is equal to the sent one
        self._cmd_execution_state['addsearchbeams'] = CmdExecState.RUNNING
        pass
        # PROTECTED REGION END #    //  CspSubarray.AddSearchBeams

    @AdminModeCheck('RemoveSearchBeamsID')
    @ObsStateCheck('removeresources')
    def is_RemoveSearchBeams_allowed(self):
        return self._is_subarray_composition_allowed()
    
    @command(
        dtype_in='DevVarUShortArray',
        doc_in="The list of SearchBeam Capability IDs to remove from the CSP sub-array.",
    )
    @DebugIt()
    @SubarrayRejectCmd(['AddSearchBeams', 'ConfigureScan'])
    def RemoveSearchBeamsID(self, argin):
        # PROTECTED REGION ID(CspSubarray.RemoveSearchBeamsID) ENABLED START #
        """
        
            Remove the specified Search Beam Capability IDs from the
            CSP sub-array.

        :param argin: 'DevVarUShortArray'
        The list of SearchBeam Capability IDs to remove from the CSP sub-array.

        :return:None
        """
        pass
        # PROTECTED REGION END #    //  CspSubarray.RemoveSearchBeamsID
        
    @AdminModeCheck('RemoveTimingBeams')
    @ObsStateCheck('removeresources')
    def is_RemoveTimingBeams_allowed(self):
        return self._is_subarray_composition_allowed()
    
    @command(
        dtype_in='DevVarUShortArray',
        doc_in="The list of Timing Beams IDs to remove from the sub-array.",
    )
    @DebugIt()
    @SubarrayRejectCmd(['AddTimingBeams', 'ConfigureScan'])
    def RemoveTimingBeams(self, argin):
        # PROTECTED REGION ID(CspSubarray.RemoveTimingBeams) ENABLED START #
        """
        Remove the specified Timing Beams from the sub-array.

        :param argin: 'DevVarUShortArray'
        The list of Timing Beams IDs to remove from the sub-array.

        :return:None
        """
        pass
        # PROTECTED REGION END #    //  CspSubarray.RemoveTimingBeams
        
    @AdminModeCheck('EndSB')
    @ObsStateCheck('endsb')
    def is_EndSB_allowed(self):
        return self._is_subarray_configuring_allowed()
    
    @command(
    )
    @DebugIt()
    @SubarrayRejectCmd(['ConfigureScan'])
    def EndSB(self):
        # PROTECTED REGION ID(CspSubarray.EndSB) ENABLED START #
        """
        Set the subarray obsState to IDLE.

        :return:None
        """
        device_list = self._sc_subarray_assigned_fqdn
        if not any(self._sc_subarray_assigned_fqdn):
            print("Assigne list is empty")
            device_list = self._sc_subarray_fqdn
        for fqdn in device_list:
            proxy = self._sc_subarray_proxies[fqdn]
            proxy.command_inout_asynch("EndSB", self._cmd_ended_cb)
        # PROTECTED REGION END #    //  CspSubarray.EndSB

# ----------
# Run server
# ----------


def main(args=None, **kwargs):
    # PROTECTED REGION ID(CspSubarray.main) ENABLED START #
    return run((CspSubarray,), args=args, **kwargs)
    # PROTECTED REGION END #    //  CspSubarray.main

if __name__ == '__main__':
    main()
