/*
 * $Id: pbsacct.c,v 1.1.1.1 2006/03/13 18:31:16 bino Exp $
 *
 * pbsacct - program to make summary and report for PBS system.
 *
 * written by:  Albino Aveleda, bino@bino.eng.br
 *
 * options:
 *   -a          : report users and groups
 *   -u          : report users
 *   -g          : report groups
 *   -b value    : cost per hour
 *   -n opt      : opt = [0..3]
 *                 0: value * cputime
 *                 1: value * walltime (default)
 *                 2: value * cputime  * ncpus per job
 *                 3: value * walltime * ncpus per job
 *   -c          : cputime information
 *   -s          : statistic information
 *   -h          : help
 */

//  Created by Fabio Roberto Vitello on 16/06/17.
//  Copyright © 2017 Fabio Roberto Vitello. All rights reserved.
//


#include <cstdlib>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "main.h"

/* global variables */
GROUP  *pgroups;
USER   *pusers;
int    f_user=0, f_group=0, f_bill=0, f_stat=0, f_cpu=0, f_stdin=0, opt=1, memorypernode,noverbose=0;
char   unit[4]={"SBU"};
double value;
int cpuvalue;

/* usage use */
void usage_exit(void)
{
    fputs("usage: pbsacct -h\n", stderr);
    exit(2);
}

/* Help */
void help()
{
    puts("pbsacct:  program to make summary and report for PBS system.");
    puts("usage:  pbsacct [-a][-u][-g][-b value][-n opt][-p value][-m value][-z][-c][-s][-h] [file(s)]");
    puts("\toptions:");
    puts("\t-a       : report per-user and per-group usage");
    puts("\t-u       : report per-user usage");
    puts("\t-g       : report per-group usage");
    puts("\t-b value : cost per hour (working with -n options)");
    puts("\t-n opt   : select how the cost is calculate");
    puts("\t           opt = [0..3] where:");
    puts("\t           0: value * cputime");
    puts("\t           1: value * walltime (default)");
    puts("\t           2: value * cputime  * ncpus per job");
    puts("\t           3: value * walltime * ncpus per job");
    puts("\t           99: Custom Option added by Fabio Vitello to compute che cost for CHIPP in MUP Cluster");
    puts("\t-c       : include information about cputime");
    puts("\t           (if your operations system support)");
    puts("\t-s       : include statistic information");
    puts("\t-p       Custom Option added by Fabio Vitello: set the number of core to compute the to compute cputime in case of exclusive node request");
    puts("\t-m       Custom Option added by Fabio Vitello: set the memory per node in GB");
    puts("\t-z       Custom Option added by Fabio Vitello: no verbose");
    puts("\t-h       : help");
    puts("\tIf there isn't input file, it use the stdin.\n");
    puts("\tExamples:");
    puts("\t- Make a statistic report per-user usage on 20 and 21");
    puts("\t  days in October, 2002.");
    puts("\t% pbsacct -u -s -p 24 -c 20021020 20021021\n");
    puts("\t- Make a report from a linux cluster in October, 2002");
    puts("\t  with informations per-user and per-group, where the");
    puts("\t  cost per hour is SBU=10.00 per CPU.");
    puts("\t% cat 200210* | pbsacct -a -p 24 -b 10.00 -n 3\n");
    exit(0);
}

/* parse args */
void parse_args(int ac, char **av)
{
    int options, err=0;
  
    /*
   f_group = 1;
    f_stat = 1;
    f_cpu = 1;
*/
    
    while ((options = getopt(ac, av, "augb:B:np:m:zsch"))!=EOF)
        switch(options) {
            case 'a':
                f_user = f_group = 1;
                break;
            case 'u':
                f_user = 1;
                break;
            case 'g':
                f_group = 1;
                break;
            case 'b':
                f_bill = 1;
                value = atof(optarg);
                if (value < 0.0) err=1;
                break;
            case 'n':
                opt = atoi(optarg);
                //99 is custom option for MUP cluster customizatio, author Fabio Vitello
                if ( ((opt < 0) && (opt > 3)) || (opt!=99) ) err=1;
                break;
            case 's':
                f_stat = 1;
                break;
            case 'c':
                f_cpu = 1;
                break;
            case 'p':
                cpuvalue = atoi(optarg);
                break;
            case 'm':
                memorypernode = atoi(optarg);
                break;
            case 'z':
                noverbose=1;
                break;
            case 'h':
                help();
                break;
            default:
                usage_exit();
                break;
        }
    if (err) usage_exit();
}

/* Main program */
int main (int argc, char **argv)
{
    int  i;
    char myname[MAX_HOSTNAME_SIZE + 1] = {"\0"};
    
    /* check if there is argument */
    parse_args(argc, argv);
    
    /* check parameters */
    if ((f_user == 0)&&(f_group == 0)){
        fputs("Please, select per-user, per-group or both usage for make report.\n",stderr);
        fputs("For help typing: ./pbsacct -h\n",stderr);
        exit(2);
    }
    if (((opt == 0)||(opt == 2))&&(f_cpu == 0)){
        fputs("Please, use '-c' option with '-n 0' and '-n 2'.\n",stderr);
        exit(2);
    }
    /* get the hostname */
    gethostname(myname, sizeof(myname));
    
    /* initialize pointers */
    pgroups = NULL;
    pusers  = NULL;
    
    /* read file(s) */
    if (optind < argc)
        for (i=optind; i < argc; i++) {
            /* printf("argv[%d]=%s\n",i,argv[i]); */
            read_file(argv[i]);
        } else {
            f_stdin++;
            read_file("\0");
        }
    
    /* print report */
    print_report(myname);
    
    /* clean memory */
    clean_groups(pgroups);
    clean_users(pusers);
}
