import http.server
import socketserver
import urllib.parse as urlparse
from urllib.parse import parse_qs
from importlib.metadata import version
import json
import sys
import traceback
import argparse
import coords


class CoordsHttpRequestHandler(http.server.SimpleHTTPRequestHandler):
    def do_GET(self):
        try:
            if self.path == '/':
                self.send_response(200)
                self.send_header('Content-type', 'text/html')
                self.end_headers()
                # Open HTML page and replace placeholders
                content = open('index.html', 'r').read()
                location_name = coords.get_location_name(location_id)
                content = content.replace('#LOCATION_NAME#', location_name)
                content = content.replace('#LIB_VERSION#', version('coordslib'))
                self.wfile.write(bytes(content, 'utf8'))
                return
            elif self.path.startswith('/coords'):
                # extract 'object' query parameter from URL
                parsed = urlparse.urlparse(self.path)
                object_name = parse_qs(parsed.query)['object'][0]
                # call coords library
                result = coords.get_coords(object_name, location_id)
                self.send_response(200)
                self.send_json_response(result)
                return
        except Exception as ex:
            print(traceback.format_exc(), file=sys.stderr)
            # send error response
            self.send_response(500)
            self.send_json_response({'error': str(ex)})
            return
        # default server GET implementation (returns file or 404)
        return http.server.SimpleHTTPRequestHandler.do_GET(self)

    def send_json_response(self, data):
        self.send_header('Content-type', 'application/json')
        self.end_headers()
        # write JSON data in HTTP response body
        self.wfile.write(bytes(json.dumps(data) + '\n', 'utf8'))


parser = argparse.ArgumentParser(description='Simple HTTP server that displays a form '
                                             'to get astronomical coordinates for a given location')
parser.add_argument('port', type=int, nargs=1, help='server port')
parser.add_argument('location_id', type=str, nargs=1, help='astropy observatory/site identifier')
args = parser.parse_args()
port = args.port[0]
location_id = args.location_id[0]

# Create an object of the above class
handler_object = CoordsHttpRequestHandler

# Avoid 'address already in use' error on server restart
socketserver.TCPServer.allow_reuse_address = True

coords_server = socketserver.TCPServer(("", port), handler_object)

# Start the server
coords_server.serve_forever()
